/**
wlkit.js - a mod merger lib
credits Ophi, dsds, wgetch, roo, vitu, basro
*/
class WLK {	
	static version = '0.53';
	static stringify = JSON.stringify;
	static ignoreFields = ['mod'];
	static textDecoder = new TextDecoder();
	static log = {log:()=>{},info:()=>{},warn:()=>{}};			
	static skipSounds = false;
	static syncHash = false;
	static parseMinifiedFunction(f){
		var a = [];
		var o = {}
		f.toString().split('\n').forEach(s=>{
			s=s.split(';')[0].split('//')[0].split('||')[0].split(')')[0].replace('Math.','');
			if(s.indexOf(' = ')==-1) return;
			s = s.split(' = ');
			if(s[1] && s[1].indexOf('.')==-1) return;			
			let k1 = s[0].split('.')[1];
			let k2 = s[1].split('.')[1];
			if(k1=='undefined'||k2=='undefined') return;
			if(k1 && k2){
				o[k1.trim()]=k2.trim();
				o[k2.trim()]=k1.trim();
			}
		});
		return o;
	}	
	static getDefinition(type,id){
		if(!WLK.defMap){
			WLK.defMap = {};
			for(let k in WLK.definitions) WLK.definitions[k].forEach(d=>WLK.defMap[d.id+'@'+k]=d);
		}
		return WLK.defMap[id+'@'+type];
	}
	static hex2int(a){
		a = parseInt(a, 16);
		if ((a & 0x8000) > 0) {
			a = a - 0x10000;
		}
		return a;
	}
	static int2hex(a){		
		let s = a<0 ? (a+65536) : a;
		s = s.toString(16);
		while(s.length<4) s='0'+s;		
		return s.toUpperCase();
	}
	static rgb2color(r, g, b, a){
		a = a >= 0 ? a : 255;
		return (((((a << 8) | r) << 8) | g) << 8) | b;
	}
	static rgb2hex(r, g, b) {
		let toh = c=>{
			let hex = (c||0).toString(16);
			return hex.length == 1 ? "0" + hex : hex;
		}
		return "#" + toh(r) + toh(g) + toh(b);
	}
	static deepCopy(o){
		return JSON.parse(WLK.stringify(o));
	}
	static liveMap = {
		wp:'weapons',
		w:'wObjects',
		n:'nObjects',
		s:'sObjects',
		spr:'sprites',
		snd:'sounds',		
		t:'textures',
		c:'constants',
		b:'behaviors',
		pale:'palette',
		conf:'config'		
	}
	static callbacks = [
		'header',
		'init',
		'update',			
		'postUpdate',
		'killed',
		'onPlayerHit',
		'onPlayerSpawn',
		'onPlayerDeath',
		'onGameStart',
		'onTick',
		'onSignal'
	]
	static __id=0;
}
class WLK_Mod {	
	constructor(zipbuffer){
		this.__id = ++WLK.__id;
		this.weapons = new WLK_Weapons();	
		this.sprites = new WLK_Sprites();		
		this.sounds = new WLK_Sounds();
		this.wObjects = new WLK_Objects('w');
		this.nObjects = new WLK_Objects('n');
		this.sObjects = new WLK_Objects('s');		
		this.textures = new WLK_Objects('t');
		this.behaviors = new WLK_Objects('b');
		this.constants = new WLK_Constants();		
		this.config = {};		
		this.palette = {};
		this.ignorize(this.config);
		this.ignorize(this.constants.obj);		
		Object.values(WLK.liveMap).forEach(v=>this[v].mod=this);
		if(zipbuffer) this.import_zip(zipbuffer);
	}	
	merge(m,opt){
		opt = opt || {
			weapons:[]			
		};
		this.merged = {};
		this.postMerge = [];
		this.postMergeBehaviors = [];
		this.postMergeBehaviorsDone = [];
		this.usedSprite = [];
		if(opt.base){
			WLK.log.info('merging base from',m.name);
			m.textures.list.forEach(s=>this.add('t',WLK.deepCopy(s)));
			m.sounds.list.forEach(s=>this.add('snd',s.clone()));
			console.info('config',m.config);
			this.config = WLK.deepCopy(m.config);
			this.constants.obj = WLK.deepCopy(m.constants.obj);
			let i;
			for(i=0;i<=265;i++) this.add('spr',m.sprites.list[i].clone());
			for(i=0;i<=13;i++) this.add('s',WLK.deepCopy(m.sObjects.list[i]));
			for(i=0;i<=23;i++) this.add('n',WLK.deepCopy(m.nObjects.list[i]));
		}
		if(!this.sprites.list.length && m.sprites.list.length>20){
			WLK.log.info('adding dummy sprite at index 0 for nObject startFrame behavior');
			this.add('spr',m.sprites.list[18].clone());
		}
		// first pass for sprites optimisation
		if(opt.mergeCheck){
			if(opt.weapons) m.weapons.list.forEach(w=>{
				if(opt.weapons.indexOf(w.name)==-1) return;
				this.mergeCheck(m,'wp',w,{force:1});
			});		
			if(opt.wObjects) m.wObjects.list.forEach(w=>{
				if(opt.wObjects.indexOf(w.id)==-1) return;
				this.mergeCheck(m,'w',w,{force:1});
			});
			WLK.log.info('DETECTED USED SPRITES', this.usedSprite);		
			if(this.usedSprite.length){
				for(let i=(opt.base?265:0);i<this.usedSprite.length;i++){
					let b = !!this.usedSprite[i];
					if(!b) continue;
					console.info('ADDING USED SPRITE AT INDEX',i);
					this.add('spr',m.sprites.list[i].clone());				
				}
			}
			this.merged = {};
			this.postMerge = [];
		}
		// real merge
		if(opt.weapons) m.weapons.list.forEach(w=>{
			if(opt.weapons.indexOf(w.name)==-1) return;
			this.mergeObject(m,'wp',w,{force:1});
		});		
		if(opt.wObjects) m.wObjects.list.forEach(w=>{
			if(opt.wObjects.indexOf(w.id)==-1) return;
			this.mergeObject(m,'w',w,{force:1});
		});
		if(this.postMergeBehaviors.length){
			WLK.log.info('taking care of behaviors',this.postMergeBehaviors);
			while(this.postMergeBehaviors.length>0){
				let obj = this.postMergeBehaviors.shift();		
				if(this.postMergeBehaviorsDone.indexOf(obj)>-1) continue;
				this.postMergeBehaviorsDone.push(obj);
				WLK.callbacks.forEach(k=>{
					let s = obj[k];
					if(!s) return;
					let a = s.match(/([WNS]ID_\d+)/g);
					if(a && a.length){
						WLK.log.info('updating behavior',obj,a);
						a.forEach(wid=>{
							let letter = wid.charAt(0).toLowerCase();
							let oldID = wid.split('_')[1]*1;
							let oldOBJ = m[WLK.liveMap[letter]].list[oldID];
							let newID = this.merged[m.__id+'_'+oldOBJ.name];
							//WLK.log.info(oldID,oldOBJ,newID);
							if(newID===undefined){
								WLK.log.info("referenced object "+wid+" in behavior has to be merged");
								this.mergeObject(m,letter,oldOBJ,{force:1});
								newID = this.merged[m.__id+'_'+oldOBJ.name];
								WLK.log.info("should be:",newID);
							}						
							s = s.split(wid).join(wid.split('_')[0]+'_$$$_'+newID);						
						});
						s = s.split('$$$_').join(''); // prevent recursive id change					
					}
					a = s.match(/(SNDID_\d+)/g);	
					if(a && a.length){
						WLK.log.info('merging sounds of behavior',obj,a);
						a.forEach(sndid=>{						
							let oldID = sndid.split('_')[1]*1;
							let oldOBJ = m.sounds.list[oldID];				
							if(oldOBJ){
								let o = this.getObjectFromHash('snd',oldOBJ.hash);		
								//WLK.log.info('oldobj',oldOBJ.hash,'match',o);				
								if(!o) {								
									o = this.add('snd',oldOBJ.clone());									
									WLK.log.info('sound was imported with id',o.id,oldOBJ);				
								}							
								s = s.split(sndid).join(sndid.split('_')[0]+'_$$$_'+o.id);						
							}else{
								s = s.split(sndid).join(sndid.split('_')[0]+'_$$$_'+sndid);
								WLK.log.info("original sound with id",oldID,"is null");
							}
							
						});
						s = s.split('$$$_').join(''); // prevent recursive id change?
					}
					obj[k]=s;					
				});
			};
		}
		this.postMerge.forEach(pm=>{
			pm[0][pm[1]]=this.merged[pm[2]];
			WLK.log.warn('postMerge',this.merged[pm[2]],pm);
		});
		this.idize(this.weapons.list);
		this.idize(this.wObjects.list);
		this.idize(this.sprites.list);
		//['w','n','s','t']
		this.computeRef();
	}
	mergeCheck(m,type,obj,opt){
		if(!this.merged) this.merged = {};
		WLK.log.log('mergeCheck',type+'id',obj.id,obj);		
		let mergeKey = m.__id+'_'+obj.name;
		if(this.merged[mergeKey]!==undefined){
			if(this.merged[mergeKey]===true) throw "oops, merge in process";			
			return this.merged[mergeKey];
		}
		this.merged[mergeKey] = true;
		//let h = obj.hash || this.hashObject(m,type,obj);
		//let deja = this.getObjectFromHash(type,h);
		opt = opt || {};
		//if(deja && !opt.force) return deja.id		
		
		obj = WLK.deepCopy(obj);
	
		let frameMap = {
			startFrame:'numFrames',
			sFrame:'rFrame',
			mFrame:'numFrames'
		}
		WLK.definitions[type].forEach(def=>{
			/*if(def.id=='rFrame' && obj[def.id]<1){
				WLK.log.log('fixed rFrame<1 for '+obj.id);
				obj[def.id]=1;
			}*/
			if((def.type=='spr') && obj[def.id]>-1){ // merge sprite/sound								
				if(type=='n' && def.id=='startFrame' && !obj[def.id]){
					WLK.log.log('special skipping startFrame of nobject');
					return;
				}				
				let idx = obj[def.id];				
				if(!frameMap[def.id]) return;//if(def.id!='startFrame' && def.id!='sFrame') return;
				if(type=='n' && idx<1) return;
				let num = obj[frameMap[def.id]] || 1;//def.id=='sFrame' ? obj.rFrame : obj.numFrames
				if(num){ // let the fun begin					
					let ha = [];
					for(let k = 0;k<=num;k++) this.usedSprite[idx+k]=1;					
				}				
				if(def.id=='startFrame' && obj.shotType>0 && obj.shotType<4){
					//WLK.log.info('importing directional frames starting at',idx,'of',obj);
					let qt = 13;
					if(obj.shotType===2) qt=15;
					for(let i=0;i<=qt;i++){
						this.usedSprite[idx+i]=obj.shotType;														
					}
				}
			}
			if(def.type.length===1 && obj[def.id]>-1){ //object ref
				let o = m[WLK.liveMap[def.type]].list[obj[def.id]];
				if(!o){
					WLK.log.warn(def,obj);
					WLK.log.warn('list',m[WLK.liveMap[def.type]].list);
					WLK.log.warn('reference =',obj[def.id]);					
					WLK.log.warn("can't get referenced object");
					obj[def.id]=-1;
					return;
				}
				let mk = m.__id+'_'+o.name;
				if(this.merged[mk]){
					if(this.merged[mk]===true){						
						WLK.log.warn('post merge add:',mk);
						//this.postMerge.push([obj,def.id,mk]);
						obj[def.id]=-2;
					}else{
						WLK.log.info('use merged ref:',this.merged[mk]);
						obj[def.id] = this.merged[mk];
					}
				}else{
					o = WLK.deepCopy(o);				
					let ex = this.getObjectFromHash(def.type,this.hashObject(m,def.type,o));
					if(ex) obj[def.id] = ex.id; // reuse existing object
					else {
						WLK.log.info('importing object',o.name);
						obj[def.id] = this.mergeCheck(m,def.type,o,{force:1});
					}
				}				
			}
		});
		//obj = {id:Math.random};//this.add(type,obj,{keepName:type=='wp',dummyw34:true});
		//obj.name = obj.id;
		this.merged[mergeKey]=obj.id;
		return obj.id;	
	}
	mergeObject(m,type,obj,opt){
		if(!this.merged) this.merged = {};
		WLK.log.log('mergeObject',type+'id',obj.id,obj);		
		let mergeKey = m.__id+'_'+obj.name;
		if(this.merged[mergeKey]!==undefined){
			if(this.merged[mergeKey]===true) throw "oops, merge in process";
			WLK.log.info('object ',mergeKey,'already merged, reusing id',this.merged[mergeKey]);
			return this.merged[mergeKey];
		}
		this.merged[mergeKey] = true;
		let h = obj.hash || this.hashObject(m,type,obj);
		let deja = this.getObjectFromHash(type,h);
		opt = opt || {};
		if(deja && !opt.force){
			WLK.log.log('OBJECT ALREADY EXIST, REUSING',type+deja.id);
			return deja.id
		}		
		obj = WLK.deepCopy(obj);
		if(type=='b'){ // behavior special merge
			this.postMergeBehaviors.push(obj);
		}
		WLK.definitions[type].forEach(def=>{
			if((def.type=='spr' || def.type=='snd') && obj[def.id]>-1){ // merge sprite/sound								
				if(type=='n' && def.id=='startFrame' && !obj[def.id]){
					WLK.log.log('special skipping startFrame of nobject');
					return;
				}
				if(def.type=='snd' && WLK.skipSounds) return;
				let idx = obj[def.id];
				let s =  m[WLK.liveMap[def.type]].list[idx];				
				if(!s) return WLK.log.warn("cannot get",idx,obj,def,m);
				let sp;
				let n0 = (type=='n' && idx<1)
				if(!n0 && def.id=='startFrame' && obj.numFrames){
					let ha = [];
					for(let k = 0;k<=obj.numFrames;k++) ha.push(m[WLK.liveMap[def.type]].list[idx+k].hash);					
					sp = this.getObjectFromHashes(def.type,ha);					
				}
				else if(!n0 && def.id=='sFrame' && obj.sFrame>-1){
					let ha = [];					
					if(obj.rFrame<0) obj.rFrame=0; 
					for(let k = 0;k<(obj.rFrame?obj.rFrame:1);k++) ha.push(m[WLK.liveMap[def.type]].list[idx+k].hash);  				
					sp = this.getObjectFromHashes(def.type,ha);					
				}
				else if(!n0 && def.id=='startFrame' && (obj.shotType>0 && obj.shotType<4)){
					let ha = [];
					let qt = 13;
					if(obj.shotType===2) qt=15;
					for(let i=0;i<=qt;i++){
						let s =  m[WLK.liveMap[def.type]].list[idx+i];
						if(s) ha.push(s.hash);
						else WLK.log.warn('cannot hash directional frame at index',idx+i,': no more sprite in the list');
						
					}										
					sp = this.getObjectFromHashes(def.type,ha);					
				}else{
					sp = this.getObjectFromHash(def.type,s.hash);
				}
				if(sp) {
					obj[def.id] = sp.id; // reuse existing object										
					return;
				}
				let o = this.add(def.type,s.clone());					
				idx = obj[def.id];
				obj[def.id] = o.id;	
				if(idx<0) return;
				if(def.id=='startSound'){					
					if(obj.numSounds>0){
						for(let i=1;i<=obj.numSounds;i++){
							let s =  m[WLK.liveMap[def.type]].list[idx+i];
							this.add(def.type,s.clone());
						}
					}				
				}
				if(def.id=='startFrame'){
					if(type=='n' && idx<1){
						WLK.log.log('skipping startFrame of nobject');
						return;
					}
					if(obj.numFrames>0){
						WLK.log.info('importing',obj.numFrames,'numFrames starting at',idx,'of',obj);
						for(let i=1;i<=obj.numFrames;i++){
							let s =  m[WLK.liveMap[def.type]].list[idx+i];
							this.add(def.type,s.clone());							
						}
					}
					if(obj.shotType>0 && obj.shotType<4){
						WLK.log.info('importing directional frames starting at',idx,'of',obj);
						let qt = 13;
						if(obj.shotType===2) qt=15;
						for(let i=1;i<=qt;i++){
							let s =  m[WLK.liveMap[def.type]].list[idx+i];
							if(s) this.add(def.type,s.clone());
							else WLK.log.warn('cannot import directional frame at index',idx+i,': no more sprite in the list');
							
						}
					}
				}
				if(def.id=='sFrame'){				
					WLK.log.info('importing',obj.rFrame,'rFrame starting at',idx,'of',obj);
					for(let i=1;i<(obj.rFrame?obj.rFrame:1);i++){
						let s =  m[WLK.liveMap[def.type]].list[idx+i];
						this.add(def.type,s.clone());							
					}
				}
			}
			if(def.type.length===1 && obj[def.id]>-1){ //object ref
				let o = m[WLK.liveMap[def.type]].list[obj[def.id]];
				if(!o){
					WLK.log.warn(def,obj);
					WLK.log.warn('list',m[WLK.liveMap[def.type]].list);
					WLK.log.warn('reference =',obj[def.id]);
					WLK.log.warn("can't get referenced object");
					obj[def.id]=-1;
					return;
				}
				let mk = m.__id+'_'+o.name;				
				if(this.merged[mk]){
					if(this.merged[mk]===true){						
						WLK.log.warn('post merge add:',mk);
						this.postMerge.push([obj,def.id,mk]);
						obj[def.id]=-2;
					}else{
						WLK.log.info('use merged ref:',this.merged[mk]);
						obj[def.id] = this.merged[mk];
					}
				}else{
					o = WLK.deepCopy(o);				
					let ex = this.getObjectFromHash(def.type,this.hashObject(m,def.type,o));
					if(ex) obj[def.id] = ex.id; // reuse existing object
					else {
						WLK.log.info('importing object',o.name);
						obj[def.id] = this.mergeObject(m,def.type,o,{force:1});
					}
				}				
			}
			
		});
		obj = this.add(type,obj,{keepName:type=='wp',dummyw34:true});
		this.merged[mergeKey]=obj.id;
		return obj.id;		
	}
	getObjectFromHash(type,hash){
		if(!hash) throw "hash is undefined";
		let list = this[WLK.liveMap[type]].list;
		let L=list.length;
		for(let i=0;i<L;i++) if((list[i].hash || this.hashObject(this,type,list[i]))==hash) return list[i];		
	}
	getObjectFromHashes(type,hashes){ // sprites/sounds only
		if(!hashes) throw "no hashes provided";
		let hash = hashes[0];		
		if(!hash) throw "no valid first hash";
		let list = this[WLK.liveMap[type]].list;
		let L=list.length;
		for(let i=0;i<L;i++){
			if(list[i].hash==hash){
				let found = list[i];
				WLK.log.log('found first hash',found);
				for(let k=1;k<hashes.length;k++){
					if(!list[i+k] || list[i+k].hash!=hashes[k]){
						found=false;
						WLK.log.log('but missmatch at index',k,hashes[k],list[i+k]);
						break;
					}
				}
				if(found) return found;
			}
		}
	}
	hashObject(m,type,obj){
		let h = '';		
		if(type=='spr' || type=='snd'){
			if(!obj.hash) WLK.log.warn('missing hash on',type,obj);
			return obj.hash;
		}		
		WLK.definitions[type].forEach(def=>{
			let v = obj[def.id];
			if(v===undefined) h+='undefined_';
			else if(def.type=='spr'||def.type=='snd'){ //sprite/sound ref
				h+=def.type;
				//WLK.log.info(h,v,m[WLK.liveMap[def.type]].list);
				if(v==-1 || (type=='n' && def.id=='startFrame' && v<1)) h+='!'; 				
				else {
					let list = m[WLK.liveMap[def.type]].list;
					if(!list[v]) {
						WLK.log.warn('reference not yet fullfiled')
						h+='??';
					}else{
						h+=list[v].hash;
					}
					
				}
				h+='_';
			}
			else if(def.type.length===1){ //object ref
				h+='ref'+obj[def.id]+'_'; //TODO minimum recursive hash?				
			}else{
				h+=''+obj[def.id]+'_';
			}
		});		
		return h;
	}
	reflect(type,obj,prop){
		let def = WLK.getDefinition(type,prop);	
		let v = obj[prop];
		if(def){
			if(def.type=='int' || def.type=='number') v = v*1;
			if(def.type=='bool') v=!!v;
			obj[prop]=v; // casting
		}
		if(!this.live) return;
		WLK.log.info('reflect',type,obj,prop);
		if(type=='conf'){
			let h = this.live.hook;
			let ka = this.live.wp.split('.');
			ka.pop();
			ka.forEach(s=>h=h[s]);			
			WLK.log.info('reflect conf',prop,h,obj);
			h[this.minMap.conf[prop]] = obj[prop];			
			return;
		}			
		if(type=='snd'){
			this.live.hooks.snd[obj.id] = obj.buffer;			
			return;
		}
		if(type=='spr' && prop=='data'){
			WLK.log.info('reflect sprite data change',type,obj,prop);
			return; // because actual data reflect will break the game
		}
		let mprop = this.minMap[type][prop];		
		let target = null;
		if(type=='c') target = this.live.hooks[type];
		else if(type=='spr') target = this.live.hooks.spr[obj.id];		
		else target = this.live.hooks[type][obj.id];
		WLK.log.info('reflecting',mprop,v,'on object',target);
		target[mprop]=v;		
		if(type=='wp' && prop=='bulletType'){ // special weapon type fix
			this.live.hooks[type][obj.id][this.live.wpt||'$d'] = this.live.hooks.w[v];
		}
		
	}	
	add(type,o,opt){	
		opt = opt || {};
		let list = this[WLK.liveMap[type]].list;
		let id = list.length;
		if(type=='w' && id==34 && opt.dummyw34){
			WLK.log.info('adding a dummy w34 to prevent unexpected behavior');
			list.push(WLK.deepCopy(WLK.WID34));
			id++;
		}
		if(type=='b') this.ignorize(o);
		o.id = id;
		//WLK.log.log('add',o,o.name);
		if(type.length===1){
			if(!o.name || (o.name.substr(0,4)==type+'id ' && !isNaN(o.name.split(' ')[1]*1))){
				o.name = type+'id '+id;				
			}
		}
		if(type=='spr') o.name='sprite '+id;
		if(type=='snd' && (!o.name || o.name.substr(0,6)=='sound ')) o.name='sound '+id;
		if(!opt.keepName){
			if(type=='wp') o.name+='2';
		}
		list.push(o);
		if(type=='spr' && !this.sprites.palette){
			//TODO there is some problem with merging mods as palette is same object ref between mods
			this.sprites.palette = o.palette;
		}
		if(!this.live) return o;
		let w;
		if(type.length<3){
			w = new this.live.hooks[type][0].constructor;
			for(let k in o) w[this.minMap[type][k]]=o[k];
			w.id = id;
			w.name = o.name;			
			this.live.hooks[type].push(w);
			if(type=='wp'){
				w[this.live.wpt||'$d'] = this.live.hooks.w[o.bulletType];
			}
		}
		if(type=='snd'){			
			this.live.hooks.snd.push(o.buffer);
		}
		if(type=='spr'){					
			w = new this.live.hooks.spr[0].constructor;		
			for(let k in o) w[this.minMap[type][k]]=o[k];
			w.width=o.width;
			w.height=o.height;
			w.data=o.data;
			this.live.hooks.spr.push(w);
		}
		WLK.log.info('reflecting mod.add on',type,o,w);
		return o;
	}
	attach(o){
		WLK.log.info('WLKit attach',o);
		this.live = o;
		this.minMap = {};
		if(this.modChangeOg !== o.hook[o.modChange]){
			this.modChangeOg = o.hook[o.modChange];
			this.modChange = o.hook[o.modChange] = p=>{
				WLK.log.info('mod changed');
				this.modChangeOg.apply(o.hook,[p]);
				this.weapons.list = 0;
				this.weapons.selected = undefined;
				this.section='wp';
				window.setTimeout(()=>{
					this.read_live(this.live);
					WLK.update && WLK.update();
				},300);
				WLK.update && WLK.update();
			}
		}
		this.read_live(o);
	}
	detach(){
		this.live.hook[this.live.modChange] = this.modChangeOg;
	}
	read_live(o){
		o.hooks = {}
		this.minMap.conf=WLK.parseMinifiedFunction(o.map.conf || o.map.mod);				
		let h = o.hook;
		let ka = o.wp.split('.');
		ka.pop();
		ka.forEach(s=>h=h[s]);		
		['colorAnim','textSpritesStartIdx','crossHairSprite','soundpack','author','name'].forEach(k=>{			
			this.config[k] = h[this.minMap.conf[k]];			
		});		
		Object.keys(WLK.liveMap).forEach(L=>{			
			if(L==='spr'){
				this.minMap[L] = {
					data:'data',
					x:o.spriteAnchor[0],
					y:o.spriteAnchor[1],
				}
			}
			if(L.length>2) return; 	
			if(L=='b') return;
			this.minMap[L]=WLK.parseMinifiedFunction(o.map[L]);			
			let h = o.hook;
			o[L].split('.').forEach(s=>h=h[s]);			
			o.hooks[L]=h;
			if(L=='c'){						
				if(h) for(let k in h) this[WLK.liveMap[L]].obj[this.minMap[L][k]]=h[k];				
				return;
			}
			this[WLK.liveMap[L]].selected=undefined;
			this[WLK.liveMap[L]].list=[];						
			let idx=0;
			if(h && h.length) h.forEach(w=>{
				let m = {};
				for(let k in this.minMap[L]) if(w[k]!==undefined) m[this.minMap[L][k]] = w[k];
				m.id=w.id===undefined ? idx : w.id;
				if(!m.name) m.name = L+'id '+(w.id===undefined?idx:w.id);				
				this[WLK.liveMap[L]].list.push(m);
				idx++;
			});
		});
		this.sprites.list = [];
		let palette = o.hook;
		o.palette.split('.').forEach(s=>palette=palette[s]);
		this.sprites.palette = palette;
		let a = o.hook;
		o.sprites.split('.').forEach(s=>a=a[s]);
		let i=0;
		o.hooks.spr = a;
		a.forEach(s=>{
			let b = new WLK_Sprite(s.width, s.height, s[o.spriteAnchor[0]]||0, s[o.spriteAnchor[1]]||0); //TODO	
			b.id = i++;
			b.name='sprite '+b.id;
			b.palette = palette;
			b.data=s.data;						
			this.sprites.list.push(b);		
		});
		this.sounds.list = [];
		a = o.sounds[0];
		a = a[o.sounds[1]];
		i=0;
		o.hooks.snd = a;
		a.forEach(s=>{			
			let snd = new WLK_Sound('',s.getChannelData(0));
			snd.id = i++;
			snd.name = 'sound '+snd.id;						
			this.sounds.list.push(snd);			
		});					
		this.computeRef();
	}	
	write_zip(opt){
		let sortName = (a,b)=>{
			if (a.name < b.name ) return -1;
			if (a.name > b.name) return 1;
			return 0;
		}
		let files = opt && opt.files ? opt.files : null;
		let zip = new JSZip();
		let log = 'mod generated with wlkit '+WLK.version+' \n';
		['name','author','version'].forEach(k=>{			
			if(this.config[k]) log+=' - '+k+': '+this.config[k]+"\n";
		});		
		log+=' - weapons: '+this.weapons.list.length;
		this.weapons.list.slice().sort(sortName).forEach(w=>log+='\n\t- '+w.name);
		if(!files || files.json || files.mod || files.json5) zip.file('mod.json5',this.write_mod());		
		if(!files || files.wlsprt || files.sprites) zip.file('sprites.wlsprt',this.write_wlsprt().buffer);
		if(!files || files.snd || files.sounds)	zip.file('sounds.snd',this.write_sounds().buffer);		
		if(!files || files.txt || files.log) zip.file("wlkit.txt", log);		
		if(!files || files.md || files.readme) if(this.config.readme) zip.file("readme.md", this.config.readme);		
		let s = zip.generate(opt);
		return s;		
	}	
	asZip(){
		return this.write_zip({type:'blob'});
	}	
	asObject(){				
		let o = {
			weapons:this.weapons.list,
			wObjects:this.wObjects.list,
			nObjects:this.nObjects.list,
			sObjects:this.sObjects.list,
			constants:this.constants.obj,
			textures:this.textures.list,
			behaviors:this.behaviors.list,
			colorAnim:this.config.colorAnim
			
		};		
		for(let k in o) o[k] = JSON.parse(WLK.stringify(o[k]));
		o.textSpritesStartIdx = this.config.textSpritesStartIdx||0;
		o.crossHairSprite = this.config.crossHairSprite||0;
		o.soundpack = this.config.soundpack||'';
		if(!o.soundpack) delete o.soundpack;
		o.name = this.config.name||'';
		o.author = this.config.author||'';
		o.version = this.config.version||'';
		return o;
	}	
	getTypeOf(o){
		/*let type=undefined;
		['w','n','s','t','wp','snd','spr'].forEach(L=>{
			this[WLK.liveMap[L]].list.forEach(o2=>{
				if(o==o2) type = L;
			});
		});*/
		let a = ['w','n','s','t','wp','snd','spr'];
		for(let i=0;i<a.length;i++){			
			if(this[WLK.liveMap[a[i]]].list.indexOf(o)!=-1) return a[i];			
		}		
	}
	computeRef(){
		let r = this.refs = {};
		this.refMap = {};
		this.refCache = {};
		['wp','w','n','s','t'].forEach(L=>this.computeRefList(L,this[WLK.liveMap[L]].list));		
		//console.info(this.refMap);
		//console.info(this.refs);
		this.computeOld();
	}
	computeRefList(type,list){
		WLK.definitions[type].forEach(def=>{
			if(def.type.length===1 || (def.type=='spr' || def.type=='snd')){
				list.forEach(o=>{
					if(o[def.id]>-1){
						let k = def.type+'_'+def.id;
						if(!this.refs[k]) this.refs[k]={};
						if(!this.refs[k][o[def.id]]) this.refs[k][o[def.id]]=[];
						if(this.refs[k][o[def.id]].indexOf(o)<0) this.refs[k][o[def.id]].push(o);
					}
				});
				if(!this.refMap[def.type]) this.refMap[def.type] = [];
				if(this.refMap[def.type].indexOf(def.id)<0) this.refMap[def.type].push(def.id);				
			}
		});
	}	
	computeOld(){		
		let r = this.ref = {};
		/* referenced in weapons */
		let bullets = r.bullets = {} //wObjects bulletType
		/* referenced in wObjects */
		let splinters = r.splinters = {} //nObjects splinterType
		let trails = r.trails = {} // sOjbects objTrailType
		let trailsO = r.trailsO = {} //nObjects partTrailObj
		let exp = r.exp = {} // sObjects createOnExp
		 /* referenced in nObjects */
		let nexp = r.nexp = {} //sObjects createOnExp
		let nsplinters = r.nsplinters = {} // nObjects splinterType
		let leaves = r.leaves = {} // sObjects leaveObj
		/* compute references */
		this.weapons.list.forEach(p=>{
			if(p.bulletType>=0){
			  if(!bullets[p.bulletType]) bullets[p.bulletType] = [];			  
			  bullets[p.bulletType].push(p); //p.name
			}
		});
		this.wObjects.list.forEach(p=>{
			if(p.splinterType>=0){
			  if(!splinters[p.splinterType]) splinters[p.splinterType] = [];			  
			  splinters[p.splinterType].push(p);
			}
			if(p.partTrailObj>=0){
			  if(!trailsO[p.partTrailObj]) trailsO[p.partTrailObj] = [];			  
			  trailsO[p.partTrailObj].push(p);
			}
			if(p.objTrailType>=0){
			  if(!trails[p.objTrailType]) trails[p.objTrailType] = [];			  
			  trails[p.objTrailType].push(p);
			}
			if(p.createOnExp>=0){
			  if(!exp[p.createOnExp]) exp[p.createOnExp] = [];			  
			  exp[p.createOnExp].push(p);
			}
		});	
		this.nObjects.list.forEach(p=>{
			if(p.createOnExp>=0){
			  if(!nexp[p.createOnExp]) nexp[p.createOnExp] = [];			  
			  nexp[p.createOnExp].push(p);
			}
			if(p.splinterType>=0){
			  if(!nsplinters[p.splinterType]) nsplinters[p.splinterType] = [];			  
			  nsplinters[p.splinterType].push(p);
			}
			if(p.leaveObj>=0){
			  if(!trailsO[p.leaveObj]) trailsO[p.leaveObj] = [];			  
			  trailsO[p.leaveObj].push(p);
			}
			if(p.objTrailType>=0){
			  if(!leaves[p.objTrailType]) leaves[p.objTrailType] = [];			  
			  leaves[p.objTrailType].push(p);
			}			
		});					
	}	
	write_mod(){
		//var p = hjson.parse(j, {keepWsc:true});
		WLK.log.log('using hjsonObject:',this.hjsonObject);
		this.computeRef();		
		var p = this.hjsonObject || hjson.parse(JSON.stringify(this.asObject()));
		['name','author','version','colorAnim','textSpritesStartIdx','crossHairSprite','soundpack'].forEach(k=>{
			p[k]=this.config[k]||'';
		});				
		if(p.behaviors) p.behaviors.forEach(b=>{
			if(b.script!==undefined){
				 delete b.script; //old cleaning fix
				console.warn('behavior had old script property on got cleaned',b);
			}
		});
		let r = this.ref;		
		function setCom(p,c){			
			if(!p.__COMMENTS__){
				Object.defineProperty(p, "__COMMENTS__", { enumerable: false, writable: true });
				p.__COMMENTS__ = {};
			}
			p.__COMMENTS__.o = [Object.keys(p)[0]];
			p.__COMMENTS__.c = {};
			c[0]='      \/\/ '+c[0];
			p.__COMMENTS__.c[p.__COMMENTS__.o[0]]=c;
		}
		function toids(a,L){
			return a.map(w => L+'id'+w.id).join(' ');	
		}
		/* add comments */		
		p.wObjects.forEach(p=>{
		  let names = [];
		  if(r.bullets[p.id]) names = r.bullets[p.id].map(w => w.name);			
		  setCom(p,["wid"+p.id+(names.length?' used in '+names.join(', '):''), ""]);
		});
		p.nObjects.forEach(p=>{
		  let s = "", k=p.id;
		  if (r.splinters[k]) s += " splinterType for "+toids(r.splinters[k],'w');
		  if (r.trailsO[k]) s += " partTrailObj for "+toids(r.trailsO[k],'w');		  
		  if (r.nsplinters[k]) s += " splinterType for "+toids(r.nsplinters[k],'n');
		  setCom(p,["nid"+k+s, ""]);		  
		});
		p.sObjects.forEach(p=>{
		  let s = "", k=p.id;
		  if (r.exp[k]) s += " createOnExp for "+toids(r.exp[k],'w');
		  if (r.trails[k]) s += " objTrailType for "+toids(r.trails[k],'w');
		  if (r.nexp[k]) s += " createOnExp for "+toids(r.nexp[k],'n');
		  if (r.leaves[k]) s += "leaveObj for "+toids(r.leaves[k],'n');
		  setCom(p,["sid"+k+s, ""]);		  
		});
		if(!p.soundpack){
			delete p.soundpack;
			if(p.__COMMENTS___) {
				delete p.__COMMENTS__.c.soundpack;
				p.__COMMENTS__.o.splice(p.__COMMENTS__.o.indexOf('soundpack'),1);
			}
		}			
		WLK.log.log(p);
		let s = hjson.stringify(p,{keepWsc:true, quotes:"all",separator:true});
		//WLK.log.log(s);
		s=s.replace('"soundpack": undefined,',''); // sometimes delete p.soundpack doesn't work!??!?!
		s=s.split('"script": undefined,').join(''); // issue seems to be with hjson
		
		return s;
  	}	
	write_json(){
		return this.write_mod();
	}
	write_wlsprt(){
		return this.sprites.asArrayBuffer();
	}
	write_sprites(){
		return this.write_wlsprt();
	}
	write_sounds(){
		return this.sounds.asArrayBuffer();
	}
	read_zip(f){
		let r = new FileReader;
		r.onload = ()=>{
			this.import_zip(r.result);
		}
		r.readAsArrayBuffer(f);
	}	
	import_zip(a){		
		let zip = new JSZip(a);
		let f = zip.file("mod.json5");
		if(!f) f = zip.file("mod.json");
		if(f) this.import_json5(f.asText());
		f = zip.file("sprites.wlsprt");
		if(f) this.import_wlsprt(f.asArrayBuffer());
		f = zip.file("sounds.snd");
		if(f) this.import_snd(f.asArrayBuffer());
		f = zip.file("readme.md");
		if(f) this.config.readme = f.asText();
		f = zip.file("LIERO.EXE") || zip.file("Liero.exe") || zip.file("liero.exe");
		if(f){
			if(!window.exe2wlmod) return WLK.log.warn('exe2wlmod.js not loaded, aborting TC zip load');
			exe2wlmod.lieroChr = undefined;
			exe2wlmod.lieroExe = f.asArrayBuffer();
			f = zip.file("LIERO.CHR") || zip.file("Liero.chr") || zip.file("liero.chr");
			if(!f) WLK.log.warn('exe2wlmod: no .chr found in zip');
			else exe2wlmod.lieroChr = f.asArrayBuffer();			
			this.import_json5(exe2wlmod.getJSON());
			this.import_wlsprt(exe2wlmod.getSprites());
			f = zip.file("LIERO.SND") || zip.file("Liero.snd") || zip.file("liero.snd");
			if(f) this.import_snd(f.asArrayBuffer());
		}
		
	}
	read_wlsprt(f){
		let r = new FileReader();
		r.onload = ()=> this.import_wlsprt(r.result);		
		r.readAsArrayBuffer(f);
	}
	read_snd(f){		
		let r = new FileReader();
		r.onload = ()=> this.import_snd(r.result);		
		r.readAsArrayBuffer(f);
	}
	read_png(f){
		let r = new FileReader();
		r.onload = ()=> this.import_png(r.result);		
		r.readAsArrayBuffer(f);
	}	
	read_json5(f){
		let r = new FileReader;
        r.onload = ()=> this.import_json5(r.result);		
        r.readAsText(f)
	}
	read_json(f) {
		this.read_json5(f);
	}
	import_png(a){
		WLK.log.info(WLK_PNG.read(a));
		//todo update palette?
		throw "unimplemented";
	}
	import_snd(a){
		if(this.importMode=='replace') this.sounds.list.length=0;
		this.sounds.read(a);
		WLK.update && WLK.update();
	}	
	import_wlsprt(a){
		if(this.importMode=='replace') this.sprites.list.length=0;
		this.sprites.read(a);
		this.computeRef();
		WLK.update && WLK.update();		
	}	
	import_json5(s){
		let a = this.hjsonObject = hjson.parse(s, {keepWsc:true});
		if(!a.behaviors) a.behaviors = [];
		this.constants.obj = a.constants;
		this.behaviors.list = a.behaviors;
		this.idize(this.behaviors.list,'b');
		this.weapons.list = a.weapons || [];
		['w','n','s'].forEach(L=>{
			this[L+'Objects'].list = a[L+'Objects'] || [];
			this.idize(this[L+'Objects'].list,L);
		});
		this.idize(this.weapons.list,'wp');		
		this.textures.list = a.textures || [];
		this.idize(this.textures.list,'t');
		this.config.name = a.name || '';
		this.config.author = a.author || '';
		this.config.version = a.version || '';
		this.config.soundpack = a.soundpack || '';
		this.config.colorAnim = a.colorAnim || [];
		this.config.textSpritesStartIdx = a.textSpritesStartIdx;
		this.config.crossHairSprite = a.crossHairSprite;		
		this.computeRef();
		if(this.postImport) this.postImport();
		WLK.update && WLK.update();
	}
	idize(a,L){
		for(let i=0;i<a.length;i++){
			a[i].id=i;
			if(L && !a[i].name) a[i].name=L+'id '+i;
			this.ignorize(a[i]);
			this.defize(a[i],L);
			if(a[i].splinterType*1===-1 && a[i].splinterAmount*1>0){
				if(!this.fixedSplinterType) this.fixedSplinterType=[];
				if(this.fixedSplinterType.indexOf(a[i].name)==-1) this.fixedSplinterType.push(a[i].name);
				a[i].splinterAmount=0;
			}
		}
	}
	defize(o,L){
		if(!L) return;
		WLK.definitions[L].forEach(def=>{
			if(def.def!==undefined && o[def.id]===undefined){
				//console.info('defizing property',def.id,'of',o,'to',def.def);
				o[def.id]=def.def;
			}
			if(def.type=='number' || def.type=='int'){
				if(typeof o[def.id] === 'string'){
					WLK.log.warn('converting string value',def.id,'to',def.type,o[def.id]);					
					let s = o[def.id];
					o[def.id]=s*1;
					if(isNaN(o[def.id])){
						WLK.log.warn('error converting to number: '+s);
						o[def.id]=s;
					}
				}
			}
		})
	}
	ignorize(o){
		WLK.ignoreFields.forEach(f=>{
			Object.defineProperty(o, f, { enumerable: false, writable: true });	
		});			
	}
}
class WLK_Constants {
	constructor(){		
		this.obj = {};
	}	
}
class WLK_Behaviors {
	constructor(){		
		this.list = [];
	}	
}
class WLK_Weapons {
	constructor(){
		this.list = [];
	}	
	getBy(prop,value){
		for(let i=0;i<this.list.length;i++){
			if(this.list[i][prop]===value) return this.list[i];
		}
	}		
	add(o){	
		let id = this.list.length;
		o.id = id;
		this.list.push(o);
		if(this.mod.live){
			let w = new this.mod.live.hooks.wp[0].constructor;
			for(let k in o) w[this.mod.minMap['wp'][k]]=o[k];
			w.id = id;
			WLK.log.info('reflecting wp add',w);
			this.mod.live.hooks.wp.push(w);			
			w[this.mod.live.wpt||'$d'] = this.mod.live.hooks.w[o.bulletType];
		}
		return o;
	}
	remove(name){
		this.list.splice(this.list.indexOf(this.getBy('name',name)),1);
	}
}
class WLK_Objects {
	constructor(type){
		this.type = type;
		this.list = [];
	}
	getByName(name){
		return this.getBy('name',name);
	}	
	getBy(prop,value){
		for(let i=0;i<this.list.length;i++){
			if(this.list[i][prop]===value) return this.list[i];
		}
	}	
}
class WLK_Sound {
	static context;
	static initContext(){
		if(!WLK_Sound.context){
			let c = WLK_Sound.context = new AudioContext;
			WLK_Sound.gainNode = c.createGain();
			WLK_Sound.gainNode.connect(c.destination);
		}
	}
	constructor(name,data){		
		//this.length = content.length;
		this.data = data;
		this.name = name;
		this.updateBuffer();		
	}
	updateBuffer(){
		WLK_Sound.initContext();
		this.buffer = WLK_Sound.context.createBuffer(1, this.data.length, 22050);
        this.buffer.getChannelData(0).set(this.data);		
		this.updateHash();
	}
	updateHash(){
		if(WLK.syncHash) this.updateHashSync();
		else this.updateHashAsync();
	}		
	updateHashSync(){
		if(WLK.skipSounds){
			this.hash=this.name||this.id||'?';
			return this.hash;
		}
		this.hash = sha256(this.data);
		return this.hash;
	}
	async updateHashAsync() {
		const hash = await crypto.subtle.digest('SHA-256', this.data);
		this.hash = WLK.textDecoder.decode(hash);		
		return this.hash;
	}
	play(){
		let b = WLK_Sound.context.createBufferSource();
        b.buffer = this.buffer;
        b.connect(WLK_Sound.gainNode);
        b.start();        
	}
	from(o){
		this.data=o.data;
		this.name=o.name;
		this.hash=o.hash;
		this.updateBuffer();
		return this;
	}
	clone(){
		let s = new WLK_Sound(this.name,this.data);		
		s.hash = this.hash;
		return s;
	}
	setWAV(b){
		WLK.log.info('setWAV',b);
		let offset=44;
		let a = new WLK_I(new DataView(b),!0);
		a.o+=offset;
		let e = new Float32Array(b.byteLength-offset);
		try{
			let f = b.byteLength-offset;
			let g = 0;
			for (; g < f; ){
				let cc = a.h(); //uint
				cc-=128;// to int
				e[g++] = cc / 2048;				
			}
			this.data = e;
			WLK.log.info('DATA',e);
			this.updateBuffer();
		}catch(er){
			WLK.log.warn(er);
			alert(er.message || er);
		}			
		
	}
	setRAW(b){
		WLK.log.info('setRAW',b);
		let a = new WLK_I(new DataView(b),!0);
		let e = new Float32Array(b.byteLength);
		try{
			let f = b.byteLength;
			let g = 0;
			for (; g < f; ){
				let cc = a.Do();
				e[g++] = cc / 2048;				
			}
			this.data = e;
			WLK.log.info('DATA',e);
			this.updateBuffer();
		}catch(er){
			WLK.log.warn(er);
			alert(er.message || er);
		}			
		
	}
	asArrayBuffer(){
		let size = this.data.length;
		let a = new Uint8ClampedArray(size);
		let c = 0, i = 0;
		var d = new DataView(a.buffer);		
		for(i=0;i<size;i++) {
			let v = Math.round(this.data[i]*2048);				
			d.setInt8(i,v,true);
		}			
		return a;
	}	
	asWAV() { //https://gist.github.com/also/900023				
		var dataSize = this.data.length;
		var buffer = new ArrayBuffer(44+dataSize);
		var dv = new DataView(buffer);
		var p = 0;
		function writeString(s) {
			for (var i = 0; i < s.length; i++) dv.setUint8(p + i, s.charCodeAt(i));			
			p += s.length;
		}
		function writeUint32(d) {
			dv.setUint32(p, d, true);
			p += 4;
		}
		function writeUint16(d) {
			dv.setUint16(p, d, true);
			p += 2;
		}
		writeString('RIFF');
		writeUint32(dataSize + 36);
		writeString('WAVE');
		writeString('fmt ');
		writeUint32(16);
		writeUint16(1);
		writeUint16(1);
		writeUint32(22050);
		writeUint32(22050);
		writeUint16(1);
		writeUint16(8);
		writeString('data');
		writeUint32(dataSize);
		for(let i=0;i<dataSize;i++) {
			let v = Math.round(this.data[i]*2048);			
			v+=128;			
			dv.setUint8(p,v,true);
			p++;
		}			
		return buffer;
	}
}
 class WLK_Sounds {
	//https://liero.nl/lierohack/docformats/liero-snd.html
	constructor(){
		this.list = [];		
	}				
	asZip(opt){
		let sortName = (a,b)=>{
			if (a.name < b.name ) return -1;
			if (a.name > b.name) return 1;
			return 0;
		}
		let zip = new JSZip();
		let log = 'soundpack generated with wlkit '+WLK.version+"\n";
		zip.file('sounds.snd',this.asArrayBuffer().buffer);		
		zip.file("wlkit.txt", log);
		let s = zip.generate(opt);
		return s;		
	}	
	asArrayBuffer(){				
		let header = (8+4+4)*this.list.length;		
		let size = 2 + header;
		this.list.forEach(s=>size+=s.data.length);		
		let a = new Uint8ClampedArray(size);
		let c = 0, i = 0;
		var d = new DataView(a.buffer);		
		d.setUint16(0,this.list.length,true); // num sounds
		c+=2;
		let offset = header+2;
		this.list.forEach(s=>{
			let L = s.data.length;
			for(i=0;i<8;i++) a[c++]=(s.name.charAt(i) || ' ').charCodeAt(0);
			d.setUint32(c,offset,true); // offset
			c+=4;
			d.setUint32(c,L,true); // length
			c+=4;						
			for(i=0;i<L;i++) {
				let v= Math.round(s.data[i]*2048);				
				d.setInt8(offset+i,v,true); //a[offset+i]=v;				
			}			
			offset+=L;
			
		});		
		return a;
		
	}
	read(a) {		
		a = new WLK_I(new DataView(a),!0);
		let b = a.la()
		  , c = []
		  , d = 0;
		
		for (; d < b; ) {
			++d;
			//a.o += 8;
			let name = new TextDecoder().decode(a.Ga(8));
			name = name.split('').map(l=>(l>='A'&&l<='Z')||(l>='0'&&l<='9')?l:' ').join('').trim();
			let e = a.Ma(); // beginning (offset) of the sound effect in the file
			let f = a.Ma(); // length of the sound effect in bytes
			let h = a.o;
			if (0 >= f)
				continue;
			a.o = e;
			e = new Float32Array(f);
			let g = 0;
			for (; g < f; ){
				let cc = a.Do();				
				e[g++] = cc / 2048;
			}
			c.push(e);
			a.o = h;
			let l = new WLK_Sound(name,e);			
			this.list.push(l);
			l.id = this.list.length-1;
			//l.name = 'sound '+l.id;
		}
		return this;
	}
	
}
class WLK_Sprites {
	constructor(){
		this.list = [];	
		this.version = 0;		
	}
	asArrayBuffer(){		
		let size = 9 + 768 + 2;
		this.list.forEach(s=>size+=8+s.width*s.height);
		let a = new Uint8ClampedArray(size);
		let c = 0, i = 0;
		var d = new DataView(a.buffer);		
		'WLSPRT'.split('').forEach(C=>a[c++]=C.charCodeAt(0));		
		d.setUint16(c++,0,true); // version
		d.setUint8(++c,1); // palette		
		for(i=0;i<768;i++) a[++c]=this.mod.sprites.palette[i];
		d.setUint16(++c,this.list.length,true); // num sprites
		c+=2; // because it's 2 bytes
		this.list.forEach(s=>{
			d.setUint16(c,s.width,true);			
			d.setUint16(c+2,s.height,true);
			d.setInt16(c+4,s.x,true);
			d.setInt16(c+6,s.y,true);
			c+=8;
			for(i=0;i<s.data.length;i++){
				a[c++]=s.data[i];
			}			
		});		
		return a;
	}
	read(a){		
		a = new WLK_I(new DataView(a),!0);
		if ("WLSPRT" != a.Je(6)) throw v.J("Invalid WLSprites File");
		this.version = a.la();
		if (0 < this.version) throw v.J("Invalid WLSprites File Version");		 
		0 != a.h() && (this.palette = new Uint8Array(a.Ga(768))); //optional palette in file
		//WLK.log.log('palette',this.palette)
		let c = a.la(), e = 0;
		for (; e < c; ) {
			++e;
			let f = a.la() // w
			  , h = a.la() // h
			  , g = a.Ic()
			  , k = a.Ic();
			if (0 == f || 0 == h) throw ("Invalid Sprite Size");
			
			let l = new WLK_Sprite(f, h, g, k);
			l.data.set(a.Ga(f * h), 0);
			l.updateHash();			
			l.palette = this.palette;
			//if(f<8 && h<8) continue;
			this.list.push(l);
			l.id = this.list.length-1;
			l.name = 'sprite '+l.id;			
		}   
		return this;
	}
}
class WLK_Sprite {	
	constructor(w, h, x, y) {
		this.width = w;
		this.height = h;
		this.x = x || 0;
		this.y = y || 0;
		this.data = new Uint8Array(w * h);
	}   
	setData(d){
		this.data=d;
		this.updateHash();		
	}
	setPNG(b){
		let r = WLK_PNG.read(b);		
		this.width = r.width;
		this.height = r.height;
		this.data = r.image;
		this.updateHash();		
	}
	updateHash(){
		if(WLK.syncHash) this.updateHashSync();
		else this.updateHashAsync();
	}
	updateHashSync(){
		this.hash = this.x+'_'+this.y+'_'+sha256(this.data);		
		return this.hash;
	}
	async updateHashAsync() {
		const hash = await crypto.subtle.digest('SHA-256', this.data);
		this.hash = this.x+'_'+this.y+'_'+WLK.textDecoder.decode(hash);
		return this.hash;
	}	
	from(o){
		this.width=o.width;
		this.height=o.height;
		this.x=o.x;
		this.y=o.y;
		this.data=o.data;
		this.hash=o.hash;
		this.palette=o.palette;
		return this;
	}
	clone(){
		let s = new WLK_Sprite(this.width,this.height,this.x,this.y);
		s.data = new Uint8Array(this.data);
		s.palette = this.palette;
		s.hash = this.hash;
		return s;
	}
	asPNG(){		
		let w=this.width, h=this.height;
		let p = new WLK_PNG(w,h,256);		
		for(let i=0;i<256;i++){
			let r = this.palette[i*3];
			let g = this.palette[i*3+1];
			let b = this.palette[i*3+2];			
			p.addPaletteColor(r,g,b,i==0 ? 0 : 255);
		}		
		for(let i=0;i<h;i++){
			for(let j=0;j<w;j++){
				let v = this.data[i*w+j];
				if(v===0) continue; //transparent
				p.buffer[p.index(j, i)] = String.fromCharCode(v);								
			}
		}		
		let dump = p.getDump();
		let a = new Array(dump.length);
		for (let i = 0; i < dump.length; i++) a[i] = dump.charCodeAt(i);
		return new Uint8Array(a);		
	}
}	
class WLK_Ga {
	static Kf(a, b) {
		return a.length <= b ? a : L.substr(a, 0, b)
	}
	static Mu(a) {
		let b = ""
		  , c = 0
		  , d = a.byteLength;
		for (; c < d; )
			b += X.yv(a[c++]);
		return b
	}
}
 class WLK_I {
	constructor(a, b) {
		null == b && (b = !1);
		this.C = a;
		this.Wa = b;
		this.o = 0
	}
	lt() {
		let a = this.bb()
		  , b = []
		  , c = 0;
		for (; c < a; ) {
			++c;
			let d = new WLK_Sa;
			d.v(this);
			b.push(d)
		}
		return b
	}
	gt() {
		let a = this.h()
		  , b = []
		  , c = 0;
		for (; c < a; ) {
			++c;
			let d = new WLK_Yb;
			d.v(this);
			b.push(d)
		}
		return b
	}
	dt() {
		let a = this.h()
		  , b = []
		  , c = 0;
		for (; c < a; ) {
			++c;
			let d = new WLK_Zb;
			d.v(this);
			b.push(d)
		}
		return b
	}
	ft() {
		let a = this.h()
		  , b = []
		  , c = 0;
		for (; c < a; ) {
			++c;
			let d = new WLK_$b;
			d.v(this);
			b.push(d)
		}
		return b
	}
	et() {
		let a = this.h()
		  , b = []
		  , c = 0;
		for (; c < a; ) {
			++c;
			let d = new WLK_ac;
			d.v(this);
			b.push(d)
		}
		return b
	}
	ht() {
		let a = this.h()
		  , b = []
		  , c = 0;
		for (; c < a; ) {
			++c;
			let d = new WLK_bc;
			d.v(this);
			b.push(d)
		}
		return b
	}
	it() {
		let a = this.h()
		  , b = []
		  , c = 0;
		for (; c < a; ) {
			++c;
			let d = new WLK_cc;
			d.v(this);
			b.push(d)
		}
		return b
	}
	Ga(a) {
		null == a && (a = this.C.byteLength - this.o);
		if (this.o + a > this.C.byteLength)
			throw v.J("Read too much");
		let b = new Uint8Array(this.C.buffer,this.C.byteOffset + this.o,a);
		this.o += a;
		return b
	}
	Co(a) {
		let b = this.Ga(a);
		a = new ArrayBuffer(a);
		(new Uint8Array(a)).set(b);
		return a
	}
	Do() {
		return this.C.getInt8(this.o++)
	}
	h() {
		return this.C.getUint8(this.o++)
	}
	Ic() {
		let a = this.C.getInt16(this.o, this.Wa);
		this.o += 2;
		return a
	}
	la() {
		let a = this.C.getUint16(this.o, this.Wa);
		this.o += 2;
		return a
	}
	j() {
		let a = this.C.getInt32(this.o, this.Wa);
		this.o += 4;
		return a
	}
	Ma() {
		let a = this.C.getUint32(this.o, this.Wa);
		this.o += 4;
		return a
	}
	M() {
		let a = this.C.getFloat32(this.o, this.Wa);
		this.o += 4;
		return a
	}
	ea() {
		let a = this.C.getFloat64(this.o, this.Wa);
		this.o += 8;
		return a
	}
	bb() {
		let a = this.o, b = 0, c, d = 0;
		for (; c = this.C.getUint8(a + b),
		5 > b && (d |= (c & 127) << 7 * b >>> 0),
		++b,
		0 != (c & 128); )
			;
		this.o += b;
		return d | 0
	}
	Je(a) {
		let b = this.o, c, d = "";
		for (a = b + a; b < a; )
			c = WLK_I.Yq(this.C, b),
			b += c.length,
			d += String.fromCodePoint(c.char);
		if (b != a)
			throw v.J("Actual string length differs from the specified: " + (b - a) + " bytes");
		this.o = b;
		return d
	}
	ec() {
		let a = this.bb();
		return 0 >= a ? null : this.Je(a - 1)
	}
	cd() {
		return this.Je(this.bb())
	}
	bi() {
		return this.Je(this.h())
	}
	ai() {
		let a = this.cd();
		return JSON.parse(a)
	}
	jt() {
		let a = this.h()
		  , b = []
		  , c = 0;
		for (; c < a; )
			++c,
			b.push(this.h());
		return b
	}
	static Yq(a, b) {
		var c = a.getUint8(b);
		let d, e, f, h, g = b;
		if (0 == (c & 128))
			++b;
		else if (192 == (c & 224))
			d = a.getUint8(b + 1),
			c = (c & 31) << 6 | d & 63,
			b += 2;
		else if (224 == (c & 240))
			d = a.getUint8(b + 1),
			e = a.getUint8(b + 2),
			c = (c & 15) << 12 | (d & 63) << 6 | e & 63,
			b += 3;
		else if (240 == (c & 248))
			d = a.getUint8(b + 1),
			e = a.getUint8(b + 2),
			f = a.getUint8(b + 3),
			c = (c & 7) << 18 | (d & 63) << 12 | (e & 63) << 6 | f & 63,
			b += 4;
		else if (248 == (c & 252))
			d = a.getUint8(b + 1),
			e = a.getUint8(b + 2),
			f = a.getUint8(b + 3),
			h = a.getUint8(b + 4),
			c = (c & 3) << 24 | (d & 63) << 18 | (e & 63) << 12 | (f & 63) << 6 | h & 63,
			b += 5;
		else if (252 == (c & 254))
			d = a.getUint8(b + 1),
			e = a.getUint8(b + 2),
			f = a.getUint8(b + 3),
			h = a.getUint8(b + 4),
			a = a.getUint8(b + 5),
			c = (c & 1) << 30 | (d & 63) << 24 | (e & 63) << 18 | (f & 63) << 12 | (h & 63) << 6 | a & 63,
			b += 6;
		else
			throw v.J("Cannot decode UTF8 character at offset " + b + ": charCode (" + c + ") is invalid");
		return {
			char: c,
			length: b - g
		}
	}
}
class WLK_D {
	constructor(a, b) {
		null == b && (b = !1);
		this.C = a;
		this.Wa = b;
		this.o = 0
	}
	oi() {
		let a = new ArrayBuffer(this.o);
		(new Uint8Array(a)).set(new Uint8Array(this.C.buffer,this.C.byteOffset,this.o));
		return a
	}
	Ob() {
		return new Uint8Array(this.C.buffer,this.C.byteOffset,this.o)
	}
	Df() {
		return new DataView(this.C.buffer,this.C.byteOffset,this.o)
	}
	il() {
		return new WLK_I(this.Df(),this.Wa)
	}
	rc(a) {
		this.C.byteLength < a && this.ut(2 * this.C.byteLength >= a ? 2 * this.C.byteLength : a)
	}
	ut(a) {
		if (1 > a)
			throw v.J("Can't resize buffer to a capacity lower than 1");
		if (this.C.byteLength < a) {
			let b = new Uint8Array(this.C.buffer);
			a = new ArrayBuffer(a);
			(new Uint8Array(a)).set(b);
			this.C = new DataView(a)
		}
	}
	su(a) {
		let b = this.o++;
		this.rc(this.o);
		this.C.setInt8(b, a)
	}
	i(a) {
		let b = this.o++;
		this.rc(this.o);
		this.C.setUint8(b, a)
	}
	hd(a) {
		let b = this.o;
		this.o += 2;
		this.rc(this.o);
		this.C.setInt16(b, a, this.Wa)
	}
	za(a) {
		let b = this.o;
		this.o += 2;
		this.rc(this.o);
		this.C.setUint16(b, a, this.Wa)
	}
	l(a) {
		let b = this.o;
		this.o += 4;
		this.rc(this.o);
		this.C.setInt32(b, a, this.Wa)
	}
	cb(a) {
		let b = this.o;
		this.o += 4;
		this.rc(this.o);
		this.C.setUint32(b, a, this.Wa)
	}
	N(a) {
		let b = this.o;
		this.o += 4;
		this.rc(this.o);
		this.C.setFloat32(b, a, this.Wa)
	}
	$(a) {
		let b = this.o;
		this.o += 8;
		this.rc(this.o);
		this.C.setFloat64(b, a, this.Wa)
	}
	eb(a) {
		let b = this.o;
		this.o += a.byteLength;
		this.rc(this.o);
		(new Uint8Array(this.C.buffer,this.C.byteOffset,this.C.byteLength)).set(a, b)
	}
	ru(a) {
		this.eb(new Uint8Array(a.buffer,a.byteOffset,a.byteLength))
	}
	si(a) {
		this.eb(new Uint8Array(a))
	}
	jd(a) {
		this.Na(WLK_D.Bh(a));
		this.vi(a)
	}
	ic(a) {
		null == a ? this.Na(0) : (this.Na(WLK_D.Bh(a) + 1),
		this.vi(a))
	}
	ui(a) {
		let b = WLK_D.Bh(a);
		if (255 < b)
			throw v.J(null);
		this.i(b);
		this.vi(a)
	}
	ti(a) {
		this.jd(JSON.stringify(a))
	}
	vi(a) {
		let b = this.o;
		this.rc(b + WLK_D.Bh(a));
		let c = a.length
		  , d = 0;
		for (; d < c; )
			b += WLK_D.gr(WLK_L.Bi(a, d++), this.C, b);
		this.o = b
	}
	Na(a) {
		let b = this.o;
		a >>>= 0;
		this.rc(b + WLK_D.Jq(a));
		this.C.setUint8(b, a | 128);
		128 <= a ? (this.C.setUint8(b + 1, a >> 7 | 128),
		16384 <= a ? (this.C.setUint8(b + 2, a >> 14 | 128),
		2097152 <= a ? (this.C.setUint8(b + 3, a >> 21 | 128),
		268435456 <= a ? (this.C.setUint8(b + 4, a >> 28 & 127),
		a = 5) : (this.C.setUint8(b + 3, this.C.getUint8(b + 3) & 127),
		a = 4)) : (this.C.setUint8(b + 2, this.C.getUint8(b + 2) & 127),
		a = 3)) : (this.C.setUint8(b + 1, this.C.getUint8(b + 1) & 127),
		a = 2)) : (this.C.setUint8(b, this.C.getUint8(b) & 127),
		a = 1);
		this.o += a
	}
	lp(a) {
		let b = 0;
		for (; b < a.length; )
			a[b++].D(this)
	}
	Gf(a) {
		if (255 < a.length)
			throw v.J(null);
		this.i(a.length);
		this.lp(a)
	}
	vu(a) {
		this.Na(a.length);
		this.lp(a)
	}
	tu(a) {
		if (255 < a.length)
			throw v.J(null);
		this.i(a.length);
		let b = 0;
		for (; b < a.length; )
			this.i(a[b++])
	}
	static ca(a, b) {
		null == b && (b = !1);
		null == a && (a = 16);
		return new WLK_D(new DataView(new ArrayBuffer(a)),b)
	}
	static gr(a, b, c) {
		let d = c;
		if (0 > a)
			throw ("Cannot encode UTF8 character: charCode (" + a + ") is negative");
		if (128 > a)
			b.setUint8(c, a & 127),
			++c;
		else if (2048 > a)
			b.setUint8(c, a >> 6 & 31 | 192),
			b.setUint8(c + 1, a & 63 | 128),
			c += 2;
		else if (65536 > a)
			b.setUint8(c, a >> 12 & 15 | 224),
			b.setUint8(c + 1, a >> 6 & 63 | 128),
			b.setUint8(c + 2, a & 63 | 128),
			c += 3;
		else if (2097152 > a)
			b.setUint8(c, a >> 18 & 7 | 240),
			b.setUint8(c + 1, a >> 12 & 63 | 128),
			b.setUint8(c + 2, a >> 6 & 63 | 128),
			b.setUint8(c + 3, a & 63 | 128),
			c += 4;
		else if (67108864 > a)
			b.setUint8(c, a >> 24 & 3 | 248),
			b.setUint8(c + 1, a >> 18 & 63 | 128),
			b.setUint8(c + 2, a >> 12 & 63 | 128),
			b.setUint8(c + 3, a >> 6 & 63 | 128),
			b.setUint8(c + 4, a & 63 | 128),
			c += 5;
		else if (-2147483648 > a)
			b.setUint8(c, a >> 30 & 1 | 252),
			b.setUint8(c + 1, a >> 24 & 63 | 128),
			b.setUint8(c + 2, a >> 18 & 63 | 128),
			b.setUint8(c + 3, a >> 12 & 63 | 128),
			b.setUint8(c + 4, a >> 6 & 63 | 128),
			b.setUint8(c + 5, a & 63 | 128),
			c += 6;
		else
			throw ("Cannot encode UTF8 character: charCode (" + a + ") is too large (>= 0x80000000)");
		return c - d
	}
	static Iq(a) {
		if (0 > a)
			throw ("Cannot calculate length of UTF8 character: charCode (" + a + ") is negative");
		if (128 > a)
			return 1;
		if (2048 > a)
			return 2;
		if (65536 > a)
			return 3;
		if (2097152 > a)
			return 4;
		if (67108864 > a)
			return 5;
		if (-2147483648 > a)
			return 6;
		throw ("Cannot calculate length of UTF8 character: charCode (" + a + ") is too large (>= 0x80000000)");
	}
	static Bh(a) {
		let b = 0
		  , c = a.length
		  , d = 0;
		for (; d < c; )
			b += WLK_D.Iq(L.Bi(a, d++));
		return b
	}
	static Jq(a) {
		a >>>= 0;
		return 128 > a ? 1 : 16384 > a ? 2 : 2097152 > a ? 3 : 268435456 > a ? 4 : 5
	}
}
class WLK_Mc {
	constructor() {}
}
class WLK_kc {
	constructor() {}
	Zm() {
		this.L = WLK_Ga.Kf(this.L, 40);
		this.Jb = WLK_Ga.Kf(this.Jb, 3)
	}
	Sd(a) {
		this.Zm();
		a.Wa = !0;
		a.za(this.Me);
		a.ui(this.L);
		a.ui(this.Jb);
		a.N(this.Zc);
		a.N(this.$c);
		a.i(this.Ub ? 1 : 0);
		a.i(this.ug);
		a.i(this.B);
		a.Wa = !1
	}
	Le(a) {
		a.Wa = !0;
		this.Me = a.la();
		this.L = a.bi();
		this.Jb = a.bi();
		this.Zc = a.M();
		this.$c = a.M();
		this.Ub = 0 != a.h();
		this.ug = a.h();
		this.B = a.h();
		a.Wa = !1;
		if (30 < this.B || 30 < this.ug)
			throw v.J(null);
		this.Zm()
	}
}
class WLK_A {
	static Cp() {
		if (null == WLK_A.yl || WLK_A.yl.buffer != WLK_A.Rb.memory.buffer)
			WLK_A.yl = new Uint8Array(WLK_A.Rb.memory.buffer);
		return WLK_A.yl
	}
	static lm(a, b) {
		b = b(a.length);
		WLK_A.Cp().set(a, b);
		WLK_A.uh = a.length;
		return b
	}
	static Of() {
		if (null == WLK_A.xl || WLK_A.xl.buffer != WLK_A.Rb.memory.buffer)
			WLK_A.xl = new Int32Array(WLK_A.Rb.memory.buffer);
		return WLK_A.xl
	}
	static Kl(a, b) {
		return WLK_A.Cp().subarray(a, a + b)
	}
	static Bx(a) {
		a = WLK_A.lm(a, WLK_A.Rb.__wbindgen_malloc);
		WLK_A.Rb.zlib_decode_raw(8, a, WLK_A.uh);
		a = WLK_A.Of()[2];
		let b = WLK_A.Of()[3]
		  , c = WLK_A.Kl(a, b).slice();
		WLK_A.Rb.__wbindgen_free(a, b);
		return c
	}
	static Mf(a) {
		a = WLK_A.lm(a, WLK_A.Rb.__wbindgen_malloc);
		WLK_A.Rb.deflate_decode_raw(8, a, WLK_A.uh);
		a = WLK_A.Of()[2];
		let b = WLK_A.Of()[3]
		  , c = WLK_A.Kl(a, b).slice();
		WLK_A.Rb.__wbindgen_free(a, b);
		return c
	}
	static Nf(a) {
		a = WLK_A.lm(a, WLK_A.Rb.__wbindgen_malloc);
		WLK_A.Rb.deflate_encode_raw(8, a, WLK_A.uh);
		a = WLK_A.Of()[2];
		let b = WLK_A.Of()[3]
		  , c = WLK_A.Kl(a, b).slice();
		WLK_A.Rb.__wbindgen_free(a, b);
		return c
	}
	static async Ad(a) {
		null == WLK_A.Rb && (WLK_A.Rb = (await WebAssembly.instantiate(await (await window.fetch(a)).arrayBuffer(), {})).instance.exports)
	}
}
WLK_A.Ad("https://www.vgm-quiz.com/dev/webliero/wledit/wasm-flate.wasm");
class WLK_PNG {
	static read(a) {		
		var b = new WLK_I(new DataView(a),!1);
		a = b.j();
		var c = b.j();
		if (-1991225785 != a || 218765834 != c)
			throw Error("File is not a png file");
		a = WLK_PNG.aq(b);
		if (1229472850 != a.type)
			throw Error("Invalid first chunk type");
		a = WLK_PNG.tw(a.data);		
		if (0 != a.Bv)
			throw Error("PNG Interlace is not supported.");
		if (3 != a.Ru)
			throw Error("Image color type is not palette.");
		if (8 != a.Ju)
			throw Error("Image bit depth should be 8");
		if (0 != a.Su || 0 != a.kv)
			throw Error("Invalid PNG image header.");
		for (c = WLK_D.ca((a.width + 1) * a.height + 1024); 0 < b.C.byteLength - b.o; ) {
			var d = WLK_PNG.aq(b);
			switch (d.type) {
			case 1229209940:
				c.eb(d.data)
			}
		}
		let pal = [];
		for(let i=0;i<768;i++){
			pal.push(b.C.getUint8(i+41,!1));	
		}			
		let ob = c.Ob();		
		b = WLK_A.Bx(ob);
		c = a.width;
		d = 0;
		for (var e = a.height; d < e; ) {
			var f = b[d++ * (c + 1)];
			if (0 != f)
				throw Error("Unsupported filter type: " + f);
		}
		d = new Uint8Array(a.width * a.height);
		e = 0;		
		for (f = a.height; e < f; ) {
			let h = e++
			  , g = 0;
			for (; g < c; ) {
				let k = g++;
				d[k + h * c] = b[1 + k + h * (c + 1)]
			}
		}
		return {
			image: d,
			palette:pal,
			width: c,
			height: a.height
		}
	}
	static tw(a) {
		a = new WLK_I(new DataView(a.buffer,a.byteOffset,a.byteLength),!1);
		return {
			width: a.Ma(),
			height: a.Ma(),
			Ju: a.h(), // \x08
			Ru: a.h(), // \x03
			Su: a.h(), // palette offset
			kv: a.h(),
			Bv: a.h()
		}
	}
	static aq(a) {
		let b = a.Ma();
		return {
			Hp: b,
			type: a.j(),
			data: a.Ga(b),
			Cx: a.Ma()
		}
	}
		
	write(buffer, offs) {
		for (var i = 2; i < arguments.length; i++) {
			for (var j = 0; j < arguments[i].length; j++) {
				buffer[offs++] = arguments[i].charAt(j);
			}
		}
	}

	byte2(w) {
		return String.fromCharCode((w >> 8) & 255, w & 255);
	}

	byte4(w) {
		return String.fromCharCode((w >> 24) & 255, (w >> 16) & 255, (w >> 8) & 255, w & 255);
	}

	byte2lsb(w) {
		return String.fromCharCode(w & 255, (w >> 8) & 255);
	}

	//https://www.xarg.org/download/pnglib.js
	constructor(width,height,depth) {
		this.width   = width;
		this.height  = height;
		this.depth   = depth;
		this.pix_size = height * (width + 1);
		this.data_size = 2 + this.pix_size + 5 * Math.floor((0xfffe + this.pix_size) / 0xffff) + 4;
		this.ihdr_offs = 0;									// IHDR offset and size
		this.ihdr_size = 4 + 4 + 13 + 4;
		this.plte_offs = this.ihdr_offs + this.ihdr_size;	// PLTE offset and size
		this.plte_size = 4 + 4 + 3 * depth + 4;
		this.trns_offs = this.plte_offs + this.plte_size;	// tRNS offset and size
		this.trns_size = 4 + 4 + depth + 4;
		this.idat_offs = this.trns_offs + this.trns_size;	// IDAT offset and size
		this.idat_size = 4 + 4 + this.data_size + 4;
		this.iend_offs = this.idat_offs + this.idat_size;	// IEND offset and size
		this.iend_size = 4 + 4 + 4;
		this.buffer_size  = this.iend_offs + this.iend_size;	// total PNG size
		this.buffer  = [];
		this.palette = {};
		this.pindex  = 0;
		this._crc32 = [];
		for (var i = 0; i < this.buffer_size; i++) this.buffer[i] = "\x00";		
		this.write(this.buffer, this.ihdr_offs, this.byte4(this.ihdr_size - 12), 'IHDR', this.byte4(width), this.byte4(height), "\x08\x03");
		this.write(this.buffer, this.plte_offs, this.byte4(this.plte_size - 12), 'PLTE');
		this.write(this.buffer, this.trns_offs, this.byte4(this.trns_size - 12), 'tRNS');
		this.write(this.buffer, this.idat_offs, this.byte4(this.idat_size - 12), 'IDAT');
		this.write(this.buffer, this.iend_offs, this.byte4(this.iend_size - 12), 'IEND');
		var header = ((8 + (7 << 4)) << 8) | (3 << 6);
		header+= 31 - (header % 31);
		this.write(this.buffer, this.idat_offs + 8, this.byte2(header));
		for (var i = 0; (i << 16) - 1 < this.pix_size; i++) {
			var size, bits;
			if (i + 0xffff < this.pix_size) {
				size = 0xffff;
				bits = "\x00";
			} else {
				size = this.pix_size - (i << 16) - i;
				bits = "\x01";
			}
			this.write(this.buffer, this.idat_offs + 8 + 2 + (i << 16) + (i << 2), bits, this.byte2lsb(size), this.byte2lsb(~size));
		}
		for (var i = 0; i < 256; i++) {
			var c = i;
			for (var j = 0; j < 8; j++) {
				if (c & 1) {
					c = -306674912 ^ ((c >> 1) & 0x7fffffff);
				} else {
					c = (c >> 1) & 0x7fffffff;
				}
			}
			this._crc32[i] = c;
		}
	}
	index(x,y) {
		var i = y * (this.width + 1) + x + 1;
		var j = this.idat_offs + 8 + 2 + 5 * Math.floor((i / 0xffff) + 1) + i;
		return j;
	}
	color(red, green, blue, alpha) {

		alpha = alpha >= 0 ? alpha : 255;
		var color = (((((alpha << 8) | red) << 8) | green) << 8) | blue;

		if (typeof this.palette[color] == "undefined") {
			WLK.log.warn('setting color without palette',red,green,blue,alpha);
			if (this.pindex == this.depth) return "\x00";

			var ndx = this.plte_offs + 8 + 3 * this.pindex;

			this.buffer[ndx + 0] = String.fromCharCode(red);
			this.buffer[ndx + 1] = String.fromCharCode(green);
			this.buffer[ndx + 2] = String.fromCharCode(blue);
			this.buffer[this.trns_offs+8+this.pindex] = String.fromCharCode(alpha);

			this.palette[color] = String.fromCharCode(this.pindex++);
		}
		return this.palette[color];
	}
	addPaletteColor(red, green, blue, alpha) {

		alpha = alpha >= 0 ? alpha : 255;
		var color = (((((alpha << 8) | red) << 8) | green) << 8) | blue;

		//if (typeof this.palette[color] == "undefined") {
			if (this.pindex == this.depth) return "\x00";

			var ndx = this.plte_offs + 8 + 3 * this.pindex;

			this.buffer[ndx + 0] = String.fromCharCode(red);
			this.buffer[ndx + 1] = String.fromCharCode(green);
			this.buffer[ndx + 2] = String.fromCharCode(blue);
			this.buffer[this.trns_offs+8+this.pindex] = String.fromCharCode(alpha);
			WLK.log.info('set palette color at index',this.pindex,red,green,blue,alpha);
			this.palette[color] = String.fromCharCode(this.pindex++);
			
		
		return this.palette[color];
	}
	// output a PNG string
	getDump(){

		// compute adler32 of output pixels + row filter bytes
		var BASE = 65521; /* largest prime smaller than 65536 */
		var NMAX = 5552;  /* NMAX is the largest n such that 255n(n+1)/2 + (n+1)(BASE-1) <= 2^32-1 */
		var s1 = 1;
		var s2 = 0;
		var n = NMAX;

		for (var y = 0; y < this.height; y++) {
			for (var x = -1; x < this.width; x++) {
				s1+= this.buffer[this.index(x, y)].charCodeAt(0);
				s2+= s1;
				if ((n-= 1) == 0) {
					s1%= BASE;
					s2%= BASE;
					n = NMAX;
				}
			}
		}
		s1%= BASE;
		s2%= BASE;
		this.write(this.buffer, this.idat_offs + this.idat_size - 8, this.byte4((s2 << 16) | s1));
		let that = this;
		// compute crc32 of the PNG chunks
		function crc32(png, offs, size) {
			var crc = -1;
			for (var i = 4; i < size-4; i += 1) {
				crc = that._crc32[(crc ^ png[offs+i].charCodeAt(0)) & 0xff] ^ ((crc >> 8) & 0x00ffffff);
			}
			that.write(png, offs+size-4, that.byte4(crc ^ -1));
		}

		crc32(this.buffer, this.ihdr_offs, this.ihdr_size);
		crc32(this.buffer, this.plte_offs, this.plte_size);
		crc32(this.buffer, this.trns_offs, this.trns_size);
		crc32(this.buffer, this.idat_offs, this.idat_size);
		crc32(this.buffer, this.iend_offs, this.iend_size);

		// convert PNG to string
		let s='';//"\211PNG\r\n\032\n"
		return "\x89PNG\r\n\x1A\n"+this.buffer.join('');
	}	
	
}
// credits https://github.com/Victorcorcos/liero-hacks/tree/main/mods
WLK.definitions = {
	"wp":[{
		"desc":"Weapon Name",
		"id":"name",
		"type":"string"
	},{
		"desc":"Type of particle shot",
		"id":"bulletType",
		"type":"w"		
	},{
		"desc":"Number of particles shot by weapon",
		"id":"parts",
		"type":"int"
	},{
		"desc":"Ammo per one magazine before reloading",
		"id":"ammo",
		"type":"int"		
	},{
		"desc":"Delay between individual shots when shooting one magazine",
		"id":"delay",
		"type":"int"
	},{
		"desc":"Reload time",
		"id":"loadingTime",
		"type":"int"
	},{
		"desc":"Recoil of shot",
		"id":"recoil",
		"type":"number"
	},{
		"desc":"Spread distribution of shot particles",
		"id":"distribution",
		"type":"number"
	},{
		"desc":"Base speed of particle. 5 is about the max playable value for usual weapons",
		"id":"bulletSpeed",
		"type":"number"
	},{
		"desc":"Percentage of worm's speed applied to the bullet",
		"id":"bulletSpeedInherit",
		"type":"number"
	},{
		"desc":"Duration of firecone sprite being displayed",
		"id":"fireCone",
		"type":"int"
	},{
		"desc":"How many shells to drop during shooting",
		"id":"leaveShells",
		"type":"int"
	},{
		"desc":"Time between dropping shells",
		"id":"leaveShellDelay",
		"type":"int"
	},{
		"desc":"Whether the weapon has the flickering, red laser sight",
		"id":"laserSight",
		"type":"bool"
	},{
		"desc":"Makes the weapon work like a laser",
		"id":"laserBeam",
		"type":"bool"
	},{
		"desc":"Sound used when shooting",
		"id":"launchSound",
		"type":"snd"
	},{
		"desc":"Play a sound when reloaded",
		"id":"playReloadSound",
		"type":"bool"
	},{
		"desc":"Custom sound used when reloaded, none for default",
		"id":"reloadSound",
		"type":"snd",
		"iif":"playReloadSound",
		"def":-1,
		"ext":true
	}	
	]
	,
	"w":[{
		"desc":"wObject Name",
		"id":"name",
		"type":"string"
	},{
		"desc":"Detect distance for particle collision",
		"id":"detectDistance",
		"type":"number"
	},{
		"desc":"Force affecting the hit worm",
		"id":"blowAway",
		"type":"number"
	},{
		"desc":"Gravity",
		"id":"gravity",
		"type":"number"
	},{
		"desc":"Spread of projectiles",
		"id":"distribution",
		"type":"number"
	},{
		"desc":"Speed added",
		"id":"addSpeed",
		"type":"number"
	},{
		"desc":"Base Speed for missile type weapon (shotType=2)",
		"id":"speed",
		"type":"number"
	},{
		"desc":"Speed multiplication on each frame",
		"id":"multSpeed",
		"type":"number"
	},{
		"desc":"Bouncyness of object",
		"id":"bounce",
		"type":"number"
	},{
		"desc":"Bouncyness friction",
		"id":"bounceFriction",
		"type":"number"
	},{
		"desc":"Sound made on explosion",
		"id":"exploSound",
		"type":"snd"
	},{
		"desc":"Which sObject to use on explosion",
		"id":"createOnExp",
		"type":"s"
	},{
		"desc":"Dirt mask type to use",
		"id":"dirtEffect",
		"type":"t"
	},{
		"desc":"Time to explode",
		"id":"timeToExplo",
		"type":"int",
		"min":0,
		"max":32767
	},{
		"desc":"Time to explode variation",
		"id":"timeToExploV",
		"type":"int"
	},{
		"desc":"Damage inflicted on hit",
		"id":"hitDamage",
		"type":"number"
	},{
		"desc":"Blood amount on hit",
		"id":"bloodOnHit",
		"type":"number"
	},{
		"desc":"Type of weapon object, see mod guide for explanation",
		"id":"shotType",
		"type":"int"
	},{
		"desc":"Color of the pixel particle",
		"id":"colorBullets",
		"type":"color"		
	},{
		"desc":"How many times to repeat physics calc per frame",
		"id":"repeat",
		"type":"int"
	},{
		"desc":"First sprite of animation",
		"id":"startFrame",
		"type":"spr"
	},{
		"desc":"Number of additional sprites to use as animation",
		"id":"numFrames",
		"type":"int"
	},{
		"desc":"Whether the animation should be looped",
		"id":"loopAnim",
		"type":"bool"
	},{
		"desc":"Whether the object should explode on worm collision",
		"id":"wormExplode",
		"type":"bool"
	},{
		"desc":"Whether the object should collide with the worm and get removed",
		"id":"wormCollide",
		"type":"bool"
	},{
		"desc":"Whether the object should explode on ground collision",
		"id":"explGround",
		"type":"bool"
	},{
		"desc":"Whether the object should collide with other objects",
		"id":"collideWithObjects",
		"type":"bool"
	},{
		"desc":"Whether the object is affected by explosions' push force",
		"id":"affectByExplosions",
		"type":"bool"
	},{
		"desc":"Number of nObjects to create on explosion",
		"id":"splinterAmount",
		"type":"int"
	},{
		"desc":"Color used on the nObjects if they don't have sprite",
		"id":"splinterColour",
		"type":"color"
	},{
		"desc":"Type of nObject used in explosion",
		"id":"splinterType",
		"type":"n"
	},{
		"desc":"Spread behavior of nObjects: 0=all directions centrifugally, 1=inherit wObject direction",
		"id":"splinterScatter",
		"type":"number"
	},{
		"desc":"sObject used as a trail",
		"id":"objTrailType",
		"type":"s"
	},{
		"desc":"Frame delay between creating trailed sObjects",
		"id":"objTrailDelay",
		"type":"int"
	},{
		"desc":"nObject trail drop type (0 crackler, 1 larpa)",
		"id":"partTrailType",
		"type":"int"
	},{
		"desc":"nObject used as a trail",
		"id":"partTrailObj",
		"type":"n"
	},{
		"desc":"Frame delay between creating trailed nObjects",
		"id":"partTrailDelay",
		"type":"int"
	},{
		"desc":"Define which team (1/2) won't take damage from this object",
		"id":"teamImmunity",
		"type":"int",
		"def":0,
		"ext":true
	},{
		"desc":"If the wObject can be exploded remotely with detonator",
		"id":"detonable",
		"type":"bool",	
		"ext":true
	},{
		"desc":"Remove this wObject when colliding with a sObject",
		"id":"removeOnSObject",
		"type":"bool",
		"ext":true
	},{
		"desc":"Prevent object to be affected by its own velocity",
		"id":"fixed",
		"type":"bool",
		"ext":true
	},{
		"desc":"Prevent object to be moved by other objects",
		"id":"immutable",
		"type":"bool",
		"ext":true
	},{
		"desc":"Make object behave like a platform that you can hook rope into",
		"id":"platform",
		"type":"bool",
		"ext":true
	},{
		"desc":"Set platform width",
		"id":"platformWidth",
		"type":"int",
		"iif":"platform",
		"ext":true
	},{
		"desc":"Set platform height",
		"id":"platformHeight",
		"type":"int",
		"iif":"platform",
		"ext":true
	},{
		"desc":"Whether platform transfer its velocity to worm",
		"id":"platformVelocityAuto",
		"type":"bool",
		"iif":"platform",
		"ext":true
	},{
		"desc":"Whether the object should draw on top of worms",
		"id":"overlay",
		"type":"bool",		
		"default":false,
		"ext":true
	},{
		"desc":"Whether the object should draw below others",
		"id":"underlay",
		"type":"bool",		
		"default":false,
		"ext":true		
	},{
		"desc":"Whether the object appears as a beacon",
		"id":"beacon",
		"type":"bool",		
		"default":false,
		"ext":true
	},{
		"desc":"Associate a behavior with custom logic",
		"id":"behavior",
		"type":"b",		
		"ext":true
	}
	],
	"n":[
	{
		"desc":"Detect distance for collision",
		"id":"detectDistance",
		"type":"number"
	},{
		"desc":"Force affecting the hit worm",
		"id":"blowAway",
		"type":"number"
	},{
		"desc":"Gravity",
		"id":"gravity",
		"type":"number"
	},{
		"desc":"Actual speed of particle, both for splinters and trails",
		"id":"speed",
		"type":"number"
	},{
		"desc":"Speed variation",
		"id":"speedV",
		"type":"number"
	},{
		"desc":"Spread of projectiles",
		"id":"distribution",
		"type":"number"
	},{
		"desc":"Bouncyness of object",
		"id":"bounce",
		"type":"number"
	},{
		"desc":"Damage inflicted on hit",
		"id":"hitDamage",
		"type":"number"
	},{
		"desc":"Whether the object should explode on worm collision",
		"id":"wormExplode",
		"type":"bool"
	},{
		"desc":"Whether the object should explode on ground collision",
		"id":"explGround",
		"type":"bool"
	},{
		"desc":"Whether the object should collide with the worm and get removed",
		"id":"wormDestroy",
		"type":"bool"
	},{
		"desc":"Blood amount on hit",
		"id":"bloodOnHit",
		"type":"number"
	},{
		"desc":"First sprite of animation",
		"id":"startFrame",
		"type":"spr"
	},{
		"desc":"Number of sprites to use as animation",
		"id":"numFrames",
		"type":"int"
	},{
		"desc":"Makes the object to be drawn onto the map and object itself gets removed from the game",
		"id":"drawOnMap",
		"type":"bool"
	},{
		"desc":"Color of the pixel particle",
		"id":"colorBullets",
		"type":"color"		
	},{
		"desc":"Which sObject to use on explosion",
		"id":"createOnExp",
		"type":"s"
	},{
		"desc":"Dirt mask type to use",
		"id":"dirtEffect",
		"type":"t"
	},{
		"desc":"Whether the object is affected by explosions' push force",
		"id":"affectByExplosions",
		"type":"bool"
	},{
		"desc":"Number of nObjects to create on explosion",
		"id":"splinterAmount",
		"type":"int"
	},{
		"desc":"Color used on the nObjects if they don't have sprite",
		"id":"splinterColour",
		"type":"color"
	},{
		"desc":"Type of object used in explosion",
		"id":"splinterType",
		"type":"n"
	},{
		"desc":"Whether object will trail blood particles",
		"id":"bloodTrail",
		"type":"bool"
	},{
		"desc":"Delay between blood particles",
		"id":"bloodTrailDelay",
		"type":"int"
	},{
		"desc":"sObject used as a trail",
		"id":"leaveObj",
		"type":"s"
	},{
		"desc":"Frame delay between creating trailed sObjects",
		"id":"leaveObjDelay",
		"type":"int"
	},{
		"desc":"Time to explode",
		"id":"timeToExplo",
		"type":"int",
		"min":0,
		"max":32767
	},{
		"desc":"Time to explode variation",
		"id":"timeToExploV",
		"type":"int"
	},{
		"desc":"Define which team (1/2) won't take damage from this object",
		"id":"teamImmunity",
		"type":"int",
		"def":0,
		"ext":true
	},{
		"desc":"Prevent object to be moved by other objects",
		"id":"immutable",
		"type":"bool",
		"ext":true
	}	
	],
	"s":[
	{
		"desc":"First sprite of animation",
		"id":"startFrame",
		"type":"spr"
	},{
		"desc":"Number of sprites to use as animation",
		"id":"numFrames",
		"type":"int"
	},{
		"desc":"Start sound",
		"id":"startSound",
		"type":"snd"
	},{
		"desc":"Num sounds",
		"id":"numSounds",
		"type":"int"
	},{
		"desc":"Delay before advancing to next frame",
		"id":"animDelay",
		"type":"int"
	},{
		"desc":"Detect range",
		"id":"detectRange",
		"type":"number"
	},{
		"desc":"Damage applied if in detect range",
		"id":"damage",
		"type":"number"
	},{
		"desc":"Force applied on the worm as push back. Can also be negative",
		"id":"blowAway",
		"type":"number"
	},{
		"desc":"Use shadow",
		"id":"shadow",
		"type":"bool"
	},{
		"desc":"Screen shake duration (not available in webliero)",
		"id":"shake",
		"type":"int"
	},{
		"desc":"Screen flash duration (not available in webliero)",
		"id":"flash",
		"type":"int"
	},{
		"desc":"Dirt mask type to use",
		"id":"dirtEffect",
		"type":"t"
	}
	],
	"c":[
	{
		"desc":"Initial length of ninja rope when shot (not sure)",
		"id":"nrInitialLength",
		"type":"number"
	},
	{
		"desc":"Minimum length for ninja rope force to apply (not sure)",
		"id":"nrAttachLength",
		"type":"number"
	},
	{
		"desc":"",
		"id":"minBounceUp",
		"type":"number"
	},
	{
		"desc":"",
		"id":"minBounceDown",
		"type":"number"
	},
	{
		"desc":"",
		"id":"minBounceHoriz",
		"type":"number"
	},
	{
		"desc":"Gravity affecting worms",
		"id":"wormGravity",
		"type":"number"
	},
	{
		"desc":"Worm acceleration",
		"id":"wormAcc",
		"type":"number"
	},
	{
		"desc":"Worm max velocity",
		"id":"wormMaxVel",
		"type":"number"
	},
	{
		"desc":"Jumping force",
		"id":"jumpForce",
		"type":"number"
	},
	{
		"desc":"max aim velocity",
		"id":"maxAimVel",
		"type":"number"
	},
	{
		"desc":"max aim acceleration",
		"id":"aimAcc",
		"type":"number"
	},
	{
		"desc":"",
		"id":"ninjaropeGravity",
		"type":"number"
	},
	{
		"desc":"in liero its length could be adjusted (victor)",
		"id":"nrMinLength",
		"type":"number"
	},
	{
		"desc":"",
		"id":"nrMaxLength",
		"type":"number"
	},
	{
		"desc":"",
		"id":"bonusGravity",
		"type":"number"
	},
	{
		"desc":"worm friction multiplier",
		"id":"wormFricMult",
		"type":"number"
	},
	{
		"desc":"aim friction",
		"id":"aimFric",
		"type":"number"
	},
	{
		"desc":"ninjarope throw velocity",
		"id":"nrThrowVel",
		"type":"number"
	},
	{
		"desc":"ninjarope pull strength",
		"id":"nrForce",
		"type":"number"
	},
	{
		"desc":"",
		"id":"bonusBounce",
		"type":"number"
	},
	{
		"desc":"",
		"id":"bonusFlickerTime",
		"type":"number"
	},
	{
		"desc":"",
		"id":"aimMaxAngle",
		"type":"number"
	},
	{
		"desc":"",
		"id":"aimMinAngle",
		"type":"number"
	},
	{
		"desc":"",
		"id":"nrAdjustVel",
		"type":"number"
	},
	{
		"desc":"",
		"id":"nrColourBegin",
		"type":"int"
	},
	{
		"desc":"",
		"id":"nrColourEnd",
		"type":"int"
	},
	{
		"desc":"",
		"id":"bonusExplodeRisk",
		"type":"number"
	},
	{
		"desc":"",
		"id":"bonusHealthVar",
		"type":"number"
	},
	{
		"desc":"",
		"id":"bonusMinHealth",
		"type":"number"
	},
	{
		"desc":"",
		"id":"firstBloodColour",
		"type":"int"
	},
	{
		"desc":"",
		"id":"numBloodColours",
		"type":"int"
	},
	{
		"desc":"gravity of worm's blood",
		"id":"bObjGravity",
		"type":"number"
	},
	{
		"desc":"",
		"id":"splinterLarpaVelDiv",
		"type":"number"
	},
	{
		"desc":"",
		"id":"splinterCracklerVelDiv",
		"type":"number"
	},
	{
		"desc":"",
		"id":"fallDamageHoriz",
		"type":"number"
	},
	{
		"desc":"",
		"id":"fallDamageDown",
		"type":"number"
	},
	{
		"desc":"",
		"id":"fallDamageUp",
		"type":"number"
	},
	{
		"desc":"",
		"id":"hFallDamage",
		"type":"bool"
	},	
	{
		"desc":"if true, weapon bonuses will only do reloads, if false - will give you a new weapon",
		"id":"hBonusReloadOnly",
		"type":"bool"
	},	
	{
		"desc":"if true, the rope will be disabled",
		"id":"noRope",
		"type":"bool",
		"ext":true
	}
	],
	"spr":[
	{
		"desc":"x anchor point",
		"id":"x",
		"type":"int"
	},{
		"desc":"y anchor point",
		"id":"y",
		"type":"int"
	}
	],
	"b":[
	{
		"desc":"behavior name",
		"id":"name",
		"type":"string"
	}
	],
	"t":[
	{
		"desc":"Whether to draw the anti-alias edges on the background (normally false for dirt and rock, true for cleaning dirt)",
		"id":"nDrawBack",
		"type":"bool"
	},
	{
		"desc":"Mask Sprite (used to cut a hole)",
		"id":"mFrame",
		"type":"spr"
	},
	{
		"desc":"Drawn Sprite (used to fill the hole)",
		"id":"sFrame",
		"type":"spr"
	},
	{
		"desc":"Number of sprites used to fill the hole, starting from sFrame",
		"id":"rFrame",
		"type":"int"
	}
	]
}
//{ UNDEF: 0, DIRT: 1, DIRT_2: 2, ROCK: 4, BG: 8, BG_DIRT: 9, BG_DIRT_2: 10, BG_SEESHADOW: 24, WORM: 32 }
WLK.materialbits = ['Undef','Dirt','Dirt2','Rock','Background','Shadow','WormM'];
// default materials
WLK.materials = [0, 9, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 32, 32, 32, 32, 32, 32, 32, 32, 32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 4, 4, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 9, 9, 9, 0, 0, 1, 1, 1, 4, 4, 4, 1, 1, 1, 4, 4, 4, 2, 2, 2, 2, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 4, 4, 4, 0, 0, 0, 0, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 24, 24, 24, 24, 8, 8, 8, 8, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]; 
// default palette
WLK.palette = [0,0,0,108,56,0,108,80,0,164,148,128,0,144,0,60,172,60,252,84,84,168,168,168,84,84,84,84,84,252,84,216,84,84,252,252,120,64,8,128,68,8,136,72,12,144,80,16,152,84,20,160,88,24,172,96,28,76,76,76,84,84,84,92,92,92,100,100,100,108,108,108,116,116,116,124,124,124,132,132,132,140,140,140,148,148,148,156,156,156,56,56,136,80,80,192,104,104,248,144,144,244,184,184,244,108,108,108,144,144,144,180,180,180,216,216,216,32,96,32,44,132,44,60,172,60,112,188,112,164,212,164,108,108,108,144,144,144,180,180,180,216,216,216,168,168,248,208,208,244,252,252,244,60,80,0,88,112,0,116,144,0,148,176,0,120,72,52,156,120,88,196,168,124,236,216,160,156,120,88,196,168,124,236,216,160,200,100,0,160,80,0,72,72,72,108,108,108,144,144,144,180,180,180,216,216,216,252,252,252,196,196,196,144,144,144,152,60,0,180,100,0,208,140,0,236,180,0,168,84,0,216,0,0,188,0,0,164,0,0,200,0,0,172,0,0,216,0,0,188,0,0,164,0,0,216,0,0,188,0,0,164,0,0,80,80,192,104,104,248,144,144,244,80,80,192,104,104,248,144,144,244,148,136,0,136,124,0,124,112,0,116,100,0,132,92,40,160,132,72,188,176,104,216,220,136,248,248,188,244,244,252,252,0,0,248,24,4,248,52,8,248,80,16,248,108,20,248,136,24,248,164,32,248,192,36,248,220,40,244,232,60,244,244,80,244,244,112,244,244,148,240,240,180,240,240,212,240,240,248,44,132,44,60,172,60,112,188,112,44,132,44,60,172,60,112,188,112,248,60,60,244,124,124,244,188,188,104,104,248,144,144,244,184,184,244,144,144,244,60,172,60,112,188,112,164,212,164,112,188,112,148,136,0,136,116,0,124,96,0,112,76,0,100,56,0,88,40,0,104,104,136,144,144,192,188,188,248,200,200,244,220,220,244,40,112,40,44,132,44,52,152,52,60,172,60,252,200,200,244,164,164,248,92,92,244,76,76,244,60,60,244,76,76,244,92,92,244,164,164,84,40,0,88,40,0,92,44,0,96,48,0,60,28,0,64,28,0,68,32,0,72,36,0,252,252,252,220,220,220,188,188,188,156,156,156,124,124,124,156,156,156,188,188,188,220,220,220,108,76,44,124,84,48,140,96,56,156,108,64,172,120,72,0,0,0,40,36,8,80,76,20,120,116,28,160,152,40,200,192,48,244,232,60,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,252,0,0,252,36,0,252,72,0,252,108,0,252,144,0,252,180,0,252,216,0,252,252,0,168,240,0,84,232,0,0,224,0,252,0,0,232,4,20,216,12,44,196,20,68,180,24,88,160,32,112,144,40,136,124,44,156,108,52,180,88,60,204,72,68,228];
// dummy wid34 for merging  
WLK.WID34 = {
	"detectDistance": 3,
	"blowAway": 0,
	"gravity": 0.02288818359375,
	"exploSound": -1,
	"addSpeed": 0,
	"distribution": 0.30517578125,
	"multSpeed": 1,
	"createOnExp": 0,
	"dirtEffect": -1,
	"wormExplode": true,
	"explGround": false,
	"wormCollide": true,
	"collideWithObjects": false,
	"affectByExplosions": true,
	"bounce": 0.30000001192092896,
	"bounceFriction": 0.800000011920929,
	"timeToExplo": 4000,
	"timeToExploV": 600,
	"hitDamage": 5,
	"bloodOnHit": 20,
	"startFrame": 210,
	"numFrames": 0,
	"loopAnim": true,
	"shotType": 0,
	"repeat": 1,
	"colorBullets": 0,
	"splinterAmount": 0,
	"splinterColour": 0,
	"splinterType": 0,
	"splinterScatter": 0,
	"objTrailType": -1,
	"objTrailDelay": 0,
	"partTrailType": 0,
	"partTrailObj": -1,
	"partTrailDelay": 0,
	"speed": 100,
	"immutable": false,
	"fixed": false,
	"platform": false,
	"teamImmunity": 0,
	"id": 34,
	"name": "wid 34 dummy",
	"removeOnSObject": true
}
// synchronous sha256, maybe we should use a faster hash function
// https://stackoverflow.com/questions/59777670/how-can-i-hash-a-string-with-sha256-in-js
var sha256 = function sha256(ascii) {
    function rightRotate(value, amount) {
        return (value>>>amount) | (value<<(32 - amount));
    };    
    var mathPow = Math.pow;
    var maxWord = mathPow(2, 32);
    var lengthProperty = 'length'
    var i, j; 
    var result = ''
    var words = [];
    var asciiBitLength = ascii[lengthProperty]*8;
    var hash = sha256.h = sha256.h || [];    
    var k = sha256.k = sha256.k || [];
    var primeCounter = k[lengthProperty];
    var isComposite = {};
    for (var candidate = 2; primeCounter < 64; candidate++) {
        if (!isComposite[candidate]) {
            for (i = 0; i < 313; i += candidate) {
                isComposite[i] = candidate;
            }
            hash[primeCounter] = (mathPow(candidate, .5)*maxWord)|0;
            k[primeCounter++] = (mathPow(candidate, 1/3)*maxWord)|0;
        }
    }
    
    ascii += '\x80'
    while (ascii[lengthProperty]%64 - 56) ascii += '\x00'
    for (i = 0; i < ascii[lengthProperty]; i++) {
        j = ascii.charCodeAt(i);
        if (j>>8) return;
        words[i>>2] |= j << ((3 - i)%4)*8;
    }
    words[words[lengthProperty]] = ((asciiBitLength/maxWord)|0);
    words[words[lengthProperty]] = (asciiBitLength)
    
    for (j = 0; j < words[lengthProperty];) {
        var w = words.slice(j, j += 16);
        var oldHash = hash;
        hash = hash.slice(0, 8);        
        for (i = 0; i < 64; i++) {
            var i2 = i + j;            
            var w15 = w[i - 15], w2 = w[i - 2];            
            var a = hash[0], e = hash[4];
            var temp1 = hash[7]
                + (rightRotate(e, 6) ^ rightRotate(e, 11) ^ rightRotate(e, 25))
                + ((e&hash[5])^((~e)&hash[6])) 
                + k[i]                
                + (w[i] = (i < 16) ? w[i] : (
                        w[i - 16]
                        + (rightRotate(w15, 7) ^ rightRotate(w15, 18) ^ (w15>>>3))
                        + w[i - 7]
                        + (rightRotate(w2, 17) ^ rightRotate(w2, 19) ^ (w2>>>10))
                    )|0
                );
            var temp2 = (rightRotate(a, 2) ^ rightRotate(a, 13) ^ rightRotate(a, 22))
                + ((a&hash[1])^(a&hash[2])^(hash[1]&hash[2])); 
            hash = [(temp1 + temp2)|0].concat(hash);
            hash[4] = (hash[4] + temp1)|0;
        }
        
        for (i = 0; i < 8; i++) {
            hash[i] = (hash[i] + oldHash[i])|0;
        }
    }
    
    for (i = 0; i < 8; i++) {
        for (j = 3; j + 1; j--) {
            var b = (hash[i]>>(j*8))&255;
            result += ((b < 16) ? 0 : '') + b.toString(16);
        }
    }
    return result;
};