/**
WebLieroEdit, a minimal mod editor
2021-12-21 initial readonly
*/
var app = angular.module('wledit',[]).config(['$sceProvider', ($sceProvider)=>{
	$sceProvider.enabled(false);
}]);
app.filter('itemreduce', function() {
    return function(list,type,filter) {		
		if(!list) return list;
		var a = [];
		list.forEach(function(o){
			if(type!='all' && o.type!=type) return;
			if(filter!='all' && o.state!=filter) return;
			a.push(o);
		});
        return a;
    };
})
app.directive("wleEscapeKey", function ($timeout){
		return function(scope, e, attrs) {
			e.on("keydown", function(ev) {				
				var keyCode = ev.which || ev.keyCode;				
				if (keyCode == 27) {					
					scope.$apply(function() { scope.$eval(attrs.wleEscapeKey); });
					ev.preventDefault();
					ev.stopPropagation(); 
				}
			});			
			$timeout(()=>{		
				e[0].focus();
			})
		};
	});
	
app.directive("wleColor", function($timeout) {
    return {
        restrict : "A",
		scope:{mod:'=',wleColor:'='},
        link:function(scope,element,att){			
			let canvas = element[0]
			function draw(){
				if(scope.wleColor==-1){
					//console.warn('WARNING COLOR is -1')
					return;
				}
				let c = scope.wleColor*3;
				let mod = scope.mod;				
				let w=canvas.width*1,h=canvas.height*1;
				let p = new Uint8ClampedArray(mod.sprites.palette);			
				canvas.width = w;
				canvas.height = h;
				canvas.style.width = w + "px";
				canvas.style.height = h + "px";
				let a = canvas.getContext("2d", {alpha:true});								
				let b = WLK.rgb2hex(p[c],p[c+1],p[c+2]);
				a.fillStyle = b;
				a.fillRect(0, 0, w, h);
			}
			draw();
			scope.$watch('wleColor', (newValue, oldValue) =>{
                if (newValue!==undefined) draw();				
            }, true);			
	
		}
    };
})
app.directive("wleSprite", function($timeout) {
    return {
        restrict : "A",
		scope:{wleSprite:'='},
        link:function(scope,element,att){			
			let canvas = element[0]
			function draw(){
				let trans = att.transparent!==undefined;
				let factor = att.factor!==undefined ? att.factor*1||1 : 1;
				var sprite = scope.wleSprite;
				let palette = new Uint8ClampedArray(sprite.palette);
				canvas.width = sprite.width;
				canvas.height = sprite.height;
				canvas.style.width = sprite.width*factor + "px";
				canvas.style.height = sprite.height*factor + "px";
				let a = canvas.getContext("2d", {alpha:true});
				a.fillStyle = trans ? "#FF000000" : "#88480c";	
				a.fillRect(0, 0, sprite.width, sprite.height);
				f = a.createImageData(sprite.width, sprite.height);
				c = f.data;
				e = d = 0;			
				let g = sprite.height;
				for (; e < g; ) {
					let k = e++
					  , l = 0
					  , m = sprite.width;
					for (; l < m; ) {
						let p = 3*sprite.data[k * sprite.width + l++];
						if(trans && !p) {
							d+=4;
						}else{
							c[d++] = palette[p];
							c[d++] = palette[p + 1];
							c[d++] = palette[p + 2];
							c[d++] = 255;
						}
					}
				}
				createImageBitmap(f).then( sp=>{				
					a.drawImage(sp, 0, 0);
				});       
			}
			draw();
			scope.$watch('wleSprite', (newValue, oldValue) =>{
                if (newValue!==undefined) draw();				
            }, true);			
			/*
			var buffer = new Uint8Array(sprite.palette.length + sprite.data.length);		
			buffer.set(sprite.palette);
			buffer.set(sprite.data, sprite.palette.length);
			e.src = URL.createObjectURL(
				new Blob([buffer], { type: 'image/png' })
			);
			//e.src = 'data:image/png;base64,'+btoa(String.fromCharCode.apply(null, scope.wleSprite.data)r);
			*/
		}
    };
})
app.directive("wleSpriteAnimated", function($timeout) {
    return {
        restrict : "A",
		scope:{wleSpriteAnimated:'=',wleSpriteAnimatedMod:'='},		
		/*
							"{mod:mod,start:mod[liveMap[mod.section]].selected[prop.id],num:mod[liveMap[mod.section]].selected.numFrames}"*/
        link:function(scope,element,att){			
			var canvas = element[0]
			var idx=0;
			var t=0;
			var destroyed=false;
			function draw(){
				if(destroyed) return;				
				let trans = att.transparent!==undefined;
				let obj = scope.wleSpriteAnimated;
				let mod = scope.wleSpriteAnimatedMod;
				let num = obj.numFrames || 0;
				if(idx>num) idx=0;			
				//console.info(obj.id,obj.startFrame,obj.startFrame+idx,idx,obj.numFrames);				
				let sprite = mod.sprites.list[obj.startFrame+idx];				
				let palette = new Uint8ClampedArray(sprite.palette);			
				canvas.width = sprite.width;
				canvas.height = sprite.height;
				canvas.style.width = sprite.width + "px";
				canvas.style.height = sprite.height + "px";
				let a = canvas.getContext("2d", {alpha:true});
				a.fillStyle = trans ? "#FF000000" : "#88480c";	
				a.fillRect(0, 0, sprite.width, sprite.height);
				f = a.createImageData(sprite.width, sprite.height);
				c = f.data;
				e = d = 0;			
				let g = sprite.height;
				for (; e < g; ) {
					let k = e++
					  , l = 0
					  , m = sprite.width;
					for (; l < m; ) {
						let p = 3*sprite.data[k * sprite.width + l++];
						if(trans && !p) {
							d+=4;
						}else{
							c[d++] = palette[p];
							c[d++] = palette[p + 1];
							c[d++] = palette[p + 2];
							c[d++] = 255;
						}
					}
				}
				createImageBitmap(f).then( sp=>{				
					a.drawImage(sp, 0, 0);
				});   				
				idx++;					
				t = setTimeout(()=>{
					t=0;
					draw()
				},300);				
			}
			draw();
			scope.$on('$destroy', ()=>{
				destroyed=true;
				clearTimeout(t);
			});
			scope.$watch('wleSpriteAnimated', (newValue, oldValue)=>{
                if (newValue!==undefined && newValue!==oldValue){					
					idx=0;
					clearTimeout(t);
					draw();				
				}
            }, true);			
		}
    };
})
app.directive("wleSpriteAnchor", function($timeout) {
    return {
        restrict : "A",
		scope:{wleSpriteAnchor:'='},
        link:function(scope,element,att){			
			let canvas = element[0]
			function draw(){
				var sprite = scope.wleSpriteAnchor;
				canvas.width = sprite.width;
				canvas.height = sprite.height;
				canvas.style.width = sprite.width + "px";
				canvas.style.height = sprite.height + "px";
				let ctx = canvas.getContext("2d", {alpha:true});
				ctx.fillStyle = "#00FFFF";	
				ctx.fillRect(sprite.x*-1, sprite.y*-1, 1, 1);				
			}
			draw();
			scope.$watch('wleSpriteAnchor', (newValue, oldValue) =>{
                if (newValue) draw();				
            }, true);						
		}
    };
})

app.directive("wleDropUpload", function($timeout) {
    return {
        restrict : "A",
		scope:{mod:'='},
        link:function(scope,element,att){			
			let extMap = {
				json:'wp',
				json5:'wp',
				wlsprt:'spr',
				snd:'snd',
				zip:'wp',
				exe:'wp',
				chr:'spr'
			}
			let e = element[0]
			e.accept = ".json,.json5,.wlsprt,.zip,.snd,.png,.exe,.chr";
            e.multiple = !0;
			e.onchange = function() {				
                if (!this.files.length) return;
				let e2m;
                for(let i=0;i<this.files.length;i++){
					let f = this.files[i];
                    let ext = "";
					let z = f.name.lastIndexOf(".");
                    -1 != z && (ext = f.name.substr(z + 1).toLowerCase());
					if(ext=='exe' || ext=='chr'){											
						let filename = f.name.toUpperCase();
						let fileReader = new FileReader();
						fileReader.onload = ()=>{							
							exe2wlmod['liero'+ext.charAt(0).toUpperCase() + ext.slice(1)] = fileReader.result;
						}
						fileReader.readAsArrayBuffer(f);
						e2m = e2m || {};
						e2m[ext]=1;
						scope.mod.loading=true;
						continue;
					}					
					let func = scope.mod['read_'+ext];
					if(!func){
						alert("unhandled file: "+ext);
						continue;
					}										
					func.apply(scope.mod,[f]);
					let s = extMap[ext];
					if(s){					
						scope.mod.section = s;
					}					
				};
				if(e2m && e2m.chr && !e2m.exe){
					scope.mod.loading=false;
					alert('you must submit a .exe file with the .chr file')
					return;
				}
				//console.info(att.wleDropUpload);
				scope.$apply(()=>scope.$eval(att.wleDropUpload))
				if(scope.mod.onFileLoad) scope.mod.onFileLoad();
				$timeout(t=>{					
					if(e2m){
						['exe','chr'].forEach(ext=>{
							if(!e2m[ext]) return;
							let func = scope.mod['import_'+(ext=='exe'?'json5':'wlsprt')];
							let conv = exe2wlmod[ext=='exe'?'getJSON':'getSprites']();
							//console.info(func,conv);
							func.apply(scope.mod,[conv]);							
						});						
						scope.mod.section = 'wp';
						scope.mod.loading=false;
						$timeout();						
					}
					let s = scope.mod.section;
					if(s!='wp' && scope.mod[WLK.liveMap[s]].list && scope.mod[WLK.liveMap[s]].list.length){
							scope.mod[WLK.liveMap[s]].selected = scope.mod[WLK.liveMap[s]].list[0];
						}
				},500);	
			}
		}
    };
})
app.directive('ace',function($timeout){
	return {
        restrict: "A",
		require: 'ngModel',		
        scope:{'aceSave':'&'},
        link: function (scope, e, att,ngModelCtrl) {
			
			var aceID = e.attr('ace-id');
			
			
			if(!window.aceIDS)  window.aceIDS = {};
			
            var editor = ace.edit(e[0]);
			window.aceIDS[aceID]=editor;
			editor.onFocus = function(){
			
			}
			var value='';
			var ready=false;
            editor.setTheme("ace/theme/twilight");
			editor.setFontSize(18);
			if(att.aceAutoScroll) editor.setAutoScrollEditorIntoView(true);
            var mode = att.aceMode || 'html';
			ngModelCtrl.$render = function() {
			
				if(ready) editor.setValue(ngModelCtrl.$viewValue || '');
				else value = ngModelCtrl.$viewValue || '';
            };						
            
            $timeout(function(){
				ready=true;				
                editor.getSession().setMode("ace/mode/"+mode);
                editor.getSession().on('change', function() {
					ngModelCtrl.$setViewValue(editor.getValue());                    
                });               
                editor.setValue(value);                
				editor.commands.addCommand({
                name: 'saveFile',
                bindKey: {
                    win: 'Ctrl-S',
                    mac: 'Command-S',
                    sender: 'editor|cli'
                },
                exec: function(env, args, request) {          
					//console.info(scope.aceSaved);				
                    scope.aceSave();                    
                }
				});
            })
            
        }
	}
	});
app.controller('wledit',($scope,$timeout,$sce)=>{	
	WLK.log = console;
	WLK.ignoreFields.push('selected');
	WLK.ignoreFields.push('$$hashKey');
	WLK.stringify = angular.toJson;
	$scope.offline = true;
	$scope.version = WLK.version;
	let randomKey = Math.random();
	$scope.trust = url => {
		$sce.trustAsResourceUrl(url);
		url+='?'+randomKey;
		return $sce.trustAsResourceUrl(url);
	}
	$scope.def = WLK.definitions;
	let liveMap = $scope.liveMap = WLK.liveMap;
	window.WLEdit = $scope.wle = {
		attach:function(o){
			console.info('WLEdit: attaching live edit') 
			$scope.live = true;
			this.attached = true
			let mod = this.mod = new WLK_Mod();
			mod.section = 'wp';
			mod.attach(o);	
			$scope.mods = [mod];
			this.closed = false;
			document.getElementById('wledit').style.display='';
			$timeout();
		}
	}
	$scope.behaviors = {
		callbacks:WLK.callbacks			
	}
	$scope.isValueError = function(prop,value){
		if(prop.min!==undefined && value*1<prop.min) return true;
		if(prop.max!==undefined && value*1>prop.max) return true;
		if((value+'').trim()==='') return true;
	}
	$scope.quickLoad = async (mod,file)=>{
		mod.loading = true;
		let hhtc = file.hhtc;
		let wlmod = file.wlmod;
		let noNewMod = file.noNewMod;
		if(hhtc) {
			mod.config.name=hhtc.name || hhtc.id.split('/').pop(); // for loading
			file.url = 'hhtc?id='+hhtc.id;
			mod.postImport = ()=>{
				mod.config.name=hhtc.name || hhtc.id.split('/').pop();
				mod.config.author=hhtc.auth;
				mod.config.readme=hhtc.desc+"\n\nArchive: "+hhtc.date;				
			}
		}
		if(wlmod) {
			mod.config.name=wlmod.name || wlmod.id.split('/').pop(); // for loading
			file.url = 'https://webliero.gitlab.io/webliero-mods/'+wlmod.id;
			mod.postImport = ()=>{
				mod.config.name=wlmod.name || wlmod.id.split('/').pop();				
			}
		}
		mod.section =  file.url && !(hhtc || wlmod) ? 'w' : 'wp';
		var req = new XMLHttpRequest();		
		if(file.url) file = file.url;
		else file = 'mods/'+file+'.zip?5';
        req.open("GET", file, true);
        req.responseType = "arraybuffer";
        req.onload = ev => {			
			mod.loading = false;
			mod.loaded = true;
			mod.import_zip(req.response);
			if(!mod.sounds.list.length && mod.weapons.list.length){
				console.info('mod has no sounds');
				let sp = mod.config.soundpack;
				if(!sp || (sp.indexOf(':')<0 && sp.indexOf('.zip')>-1)){
					console.info('loading soundpack:',sp || 'default');
					$scope.loadWLSP(mod,sp);
				}				
					
			}
			let m = mod[WLK.liveMap[mod.section]];
			if(mod.section!='wp' && m.list && m.list.length) m.selected = m.list[0];			
			if(!noNewMod) $scope.newMod();
			$timeout();
        };
        req.send(null);	
	}
	$scope.loadWLSP = (mod,f)=>{
		var req = new XMLHttpRequest();		
		let url = f ? "wlsp?id="+f : 'sp/default.zip';
		req.open("GET", url, true);
        req.responseType = "arraybuffer";
        req.onload = ev => {						
			mod.import_zip(req.response);
			console.info('downloaded soundpack '+f+' for consistency');
			$timeout();
        };
        req.send(null);	
	}
	$scope.close = ()=>{
		$scope.mods[0].detach();
		$scope.mods = [];
		window.WLEdit.closed=true;
		document.getElementById('wledit').style.display='none';
		$timeout();
	}	
	
	$scope.getAffectedWeapons = function(mod){
		if(!mod.affectedWeapons) mod.affectedWeapons = [];
		mod.affectedWeapons.length=0;
		let done = [];
		let src = ['splinterType','partTrailObj','createOnExp','dirtEffect','leaveObj','objTrailType'];
		let a = [mod[liveMap[mod.section]].selected];
		while(a.length){
			let o = a.shift();				
			let t = mod.getTypeOf(o);
			done.push(o);
			mod.refMap[t].forEach(c=>{
				console.info(c);
				let refs = mod.refs[t+'_'+c];
				if(!refs){
					console.warn("cant get refs",t,c,o.id,mod)
					return;
				}
				refs = refs[o.id];			
				if(!refs){
					console.warn("cant get object in refs",t,c,o.id,mod)
					return;
				}
				if(!refs.length) return;
				refs.forEach(r=>{
					if(src.indexOf(c)!=-1 && done.indexOf(r)<0 && a.indexOf(r)<0) a.push(r);
					if(c=='bulletType'){
						if(mod.affectedWeapons.indexOf(r)<0) mod.affectedWeapons.push(r);
					}
					console.info('r',r);
				});
			});		
		}
		return mod.affectedWeapons;
	}
	WLK.update = $timeout; // force angular digest when mod is modified
	$scope.mods = [];
	$scope.newMod = ()=>{		
		let mod = new WLK_Mod();
		mod.selectedCallback = 'update';//default behavior cb
		mod.newMod=$scope.newMod;//....
		mod.onFileLoad = p=>{
			mod.loaded = true;
			$scope.live = true;
			mod.onFileLoad=undefined;
		}	
		$scope.mods.push(mod);
		return mod;
	}
	$scope.newMod();
	
	
	if(window.onWLEdit) window.onWLEdit(WLEdit);	
	$scope.forceRefresh = function(){
		window.onWLEdit(WLEdit);
	}
	$scope.setLiveSection = (mod,type,idx)=>{
		mod.section=type;		
		if(idx!==undefined) mod[liveMap[type]].selected = mod[liveMap[type]].list[idx];
	}
	
	$scope.changeObjectRef = (mod,prop,obj)=>{			
		var r = prompt('Change '+(prop.type.length===1?prop.type+'id':prop.type),obj[prop.id])*1;
		if(isNaN(r)) return;
		obj[prop.id] = r;
		console.info('changing',prop.id,'to',r);
		mod.reflect(mod.section,obj,prop.id);
		
	}	
	
	$scope.onSectionChange = mod=>{
		if(mod.section=='c'){
			mod.constants.selected = mod.constants.obj;
		}
		if(mod.section=='conf'){
			mod.config.selected = mod.config;
		}	
	}
	$scope.copyToMod = (mod,dest)=>{
		console.info('copyToMod',mod,dest);
		if(mod.section=='snd' || mod.section=='spr'){
			dest.section=mod.section;
			$scope.addNew(dest,mod[liveMap[mod.section]].selected);
		}		
		if(mod.section=='wp'){			
			dest.mergeObject(mod,'wp',mod.weapons.selected);
			dest.section='wp';			
			dest.weapons.selected = dest.weapons.list[dest.weapons.list.length-1];
			dest.computeRef();
		}
	}	
	$scope.replaceToMod = (mod,dest)=>{
		console.info('replaceToMod',mod,dest);
		let t = dest[liveMap[dest.section]].selected;
		if(!t) return alert('no target object selected')
		let o = mod[liveMap[mod.section]].selected;
		if(o.clone) t.from(o.clone());
		else alert('not implemented');		
	}
	$scope.deleteWeapon = (mod)=>{	
		if(mod.section!='wp') return;
		let wp = mod.weapons.selected;
		if(!wp) return;
		if(!confirm("Delete weapon "+wp.name+"?\nThis will not delete orphaned objects")) return;
		let a = mod.weapons.list;
		a.splice(a.indexOf(wp),1);
		mod.weapons.selected = undefined;		
	}
	$scope.deepCopyWeapon = (mod,ev)=>{			
		if(!ev.ctrlKey) alert("feature not yet available sorry");		
	}
	$scope.onSpriteAnchorChange = (mod,p)=>{
		mod.sprites.selected[p]=mod.sprites.selected[p]*1||0;
		mod.reflect(mod.section,mod.sprites.selected,p);
	}
	
	$scope.onModConfChange = (mod,prop)=>{		
		if(prop=='textSpritesStartIdx' || prop=='crossHairSprite') mod.config[prop]=mod.config[prop]*1;
		mod.reflect(mod.section,mod.config,prop);
	}
	
	$scope.addNew = (mod,o)=>{				
		o = o || mod[liveMap[mod.section]].selected || mod[liveMap[mod.section]].list[0];
		if(!o){
			if(mod.section=='b') o = {script:''}
			if(!o) return alert("no initial object to clone!");
		}
		if(o.clone) o = o.clone();
		else o = WLK.deepCopy(o);
		console.info('addNew',o);
		mod[liveMap[mod.section]].selected = mod.add(mod.section,o);
	}
	function askFile(ext,cb){
		let e=document.createElement('input');
		document.getElementById('wledit').appendChild(e);
		e.type="file";
		setTimeout(function(){
			e.click();
		},100);				
		e.accept = ext;
        e.multiple = 0;
		e.onchange = ev=>{
			document.getElementById('wledit').removeChild(e);
			if (!e.files.length) return;
			let r = new FileReader();
			r.onload = () => {
				cb(r.result,e.files[0]);
			}
			r.readAsArrayBuffer(e.files[0]);			
		}
		e.style.display="none";
		e.style.opacity=0;
		e.style.height="1px";
	}
	$scope.changeColorOf = (mod,obj,prop)=>{		
		let c = obj[prop];		
		if($scope.colorSelect) $scope.colorSelect.close();
		$scope.colorSelect = {
			list:[],
			mod:mod,
			select:function(c){				
				obj[prop]=c;
				$scope.colorSelect.close();
				mod.reflect(mod.section,obj,prop);
			},
			close:function(){
				if(!$scope.colorSelect) return;				
				window.removeEventListener("mouseup",$scope.colorSelect.close);
				$scope.colorSelect=0;
				$timeout();
			}
			
		}
		window.addEventListener("mouseup",$scope.colorSelect.close);
		for(let i=0;i<256;i++) $scope.colorSelect.list.push(i);
	}
	$scope.changeColorAnim = mod=>{
		let s = mod.config.colorAnim.join(',');
		let r = prompt("comma separated values",s);
		if(!r || r==s) return;
		let a = mod.config.colorAnim;
		a.length=0;		
		r.split(',').forEach(v=>a.push(v.trim()*1||0));		
	}
	$scope.selectConversionChoice = function(sprite,choice){
		sprite.choices = null;
		sprite.width = choice.width;
		sprite.height = choice.height;
		sprite.setData(choice.data);		
	}
	$scope.changeSpritePNG = mod=>{
		askFile('.png', (r,f)=>{			
			let s = mod.sprites.selected;
			try {
				//console.info(f,r);
				try {
					s.setPNG(r);	
				}catch(ex){
					//console.warn(ex);					
					if(!confirm("Error: "+ex.message+"\nThe image is probably not liero palette indexed.\nDo you want to try automatic palette conversion?")) return;
					let algos = [simpleColorDistance,deltaE];
					
					let img = document.createElement("img");
					let p = new Uint8ClampedArray(mod.sprites.palette);
					let pal = [];					
					for(let i=0;i<256;i++){
						let c = i*3;
						pal.push([p[c],p[c+1],p[c+2]]);
					}			
								
					img.onload = () => {
						s.choices = [];		
						try {				
						let canvas = document.createElement("canvas");
						let context = canvas.getContext('2d');						
						var w = img.naturalWidth, h = img.naturalHeight;						
						context.drawImage(img,0,0);
						var imgd = context.getImageData(0, 0, w, h);
						var pix = imgd.data;						
						URL.revokeObjectURL(this.src);
						algos.forEach(algo=>{							
							let bests = {};
							let data = new Uint8Array(w * h);
							// loop over each pixel and find best match in palette.
							let c=0;
							for (let i = 0, n = pix.length; i < n; i += 4) {							
								let rgb = [pix[i],pix[i+1],pix[i+2]];								
								let k = rgb[0]+'-'+rgb[1]+'-'+rgb[2];								
								if(bests[k]===undefined){								
									let best=101;
									let idx=-1;
									for(let j=0;j<256;j++){
										let v = algo(rgb,pal[j]);										
										if(v<best){
											best=v;
											idx=j;
										}
									}			
									bests[k] = idx;									
								}
								data[c++] = bests[k]								
							}							
							let tmps = new WLK_Sprite(w,h,s.x,s.y);
							tmps.palette = s.palette;							
							tmps.setData(data);							
							tmps.algo = algo.toString().split('(')[0].split(' ')[1];
							s.choices.push(tmps);				
						});
						$timeout();
						}catch(err){
							console.warn(err);
							alert(err.message);
							return;
						}						
					}
					img.src = URL.createObjectURL(f);										
				}
				if(f && f.name){
					let a = f.name.split('.')[0].split('_');
					a.shift();
					if(a.length==2 && a[0].length==4 && a[1].length==4){
						console.info('importing anchor point from wltool filename',a);
						s.x = WLK.hex2int('0x'+a[0]);
						s.y = WLK.hex2int('0x'+a[1]);
					}
					
				}
			}catch(e){
				alert(e.message || e);
				return;
			}			
			mod.sprites.selected=null;
			$timeout(()=>{ 
				mod.sprites.selected=s;
				//mod.reflect(mod.section,mod.sprites.selected,'data');
				//if($scope.wle.attached) alert("To see in-game changes, export .wlsprt and re-import it with /loadmod");

			},100);
			$timeout();
		});		
	}
	$scope.importPalette = (mod,type)=>{
		askFile('.'+type, r=>{									
			let a = [];
			if(type=='json'){
				a = JSON.parse(new TextDecoder().decode(r));				
			}else if(type=='lpl'){
				a = new Uint8ClampedArray(r);				
			}else if(type=='png'){
				let png = WLK_PNG.read(r);
				a = png.palette;
			}		
			if(!a || !a.length) return alert('invalid array');
			let p = mod.sprites.palette;
			for(let i=0;i<768;i++){
				if(p[i]!=a[i]){
					console.info('changing palette at index',i,'from',p[i],'to',a[i]);
				}
				p[i]=a[i];
			}
			$timeout();
		});	
		
	}
	$scope.downloadPalette = (mod,type)=>{
		let pal = mod.sprites.palette;
		if(type=='json'){
			let a = [];
			for(let i=0;i<768;i++) a.push(pal[i]);			
			let b = new Blob([JSON.stringify(a)], {type: "text/plain;charset=utf-8"});
			window.saveAs(b, 'palette.json', {type:'application/json'});
			return;
		}
		if(type=='lpl'){
			let a = new Uint8ClampedArray(768);
			for(let i=0;i<768;i++) a[i] = pal[i];
			let b = new Blob([a.buffer]);
			window.saveAs(b, 'palette.lpl', {type:'application/octet-stream'});
			return;
		}		
		let w=256, h=1;
		let p = new WLK_PNG(w,h,256);		
		for(let i=0;i<256;i++){
			let r = pal[i*3];
			let g = pal[i*3+1];
			let b = pal[i*3+2];			
			p.addPaletteColor(r,g,b,i==0 ? 0 : 255);
		}		
		for(let i=0;i<w;i++){
			p.buffer[p.index(i, 0)] = String.fromCharCode(i);
		}		
		let dump = p.getDump();
		let a = new Array(dump.length);
		for (let i = 0; i < dump.length; i++) a[i] = dump.charCodeAt(i);
		let b = new Uint8Array(a);					
		b = new Blob([b]);
		window.saveAs(b, 'palette.png', {type:'image/png'});		
	}
	$scope.downloadSpritePNG = (sprite)=>{
		let b = sprite.asPNG();
		console.info(sprite,b);
		b = new Blob([b]);
		window.saveAs(b, sprite.id+'_'+WLK.int2hex(sprite.x)+'_'+WLK.int2hex(sprite.y)+'.png', {type:'image/png'});		
	}
	$scope.changeSoundWAV = (mod,ext)=>{
		askFile(ext||'.wav', r=>{			
			let s = mod.sounds.selected;
			try {
				if(ext=='.wav') s.setWAV(r);
				else s.setRAW(r);
			}catch(e){
				alert(e.message || e);
				return;
			}			
			mod.sounds.selected=null;
			$timeout(()=>{ 
				mod.sounds.selected=s;
				mod.reflect(mod.section,mod.sounds.selected,'data');
			},100);
			$timeout();
		});		
	}
	$scope.downloadWAV = sound=>{		
		let b = sound.asWAV();
		console.info(sound,b);
		b = new Blob([b]);
		window.saveAs(b, sound.name+'.wav', {type:'application/octet-stream'});		
	}	
	$scope.downloadRAW = sound=>{		
		
		let b = sound.asArrayBuffer();
		console.info(sound,b);
		b = new Blob([b]);
		window.saveAs(b, sound.name+'.wlsnd', {type:'application/octet-stream'});		
	}
	$scope.download = (mod,type)=>{		
		let t = {
			json:['json','mod.json5'],
			zip:['zip','mod.zip'],
			sprites:['octet-stream','sprites.wlsprt'],
			sounds:['octet-stream','sounds.snd']
		}		
		let b = mod['write_'+type]({type:'blob'});
		if(type=='json') b = new Blob([b], {type: "text/plain;charset=utf-8"});
		if(b instanceof Uint8ClampedArray) b = b.buffer;
		if(b instanceof ArrayBuffer) b = new Blob([b]);
		let name = t[type][1];
		if(name=='mod.zip' && mod.config.name) name=mod.config.name+(mod.config.version?'_'+mod.config.version:'')+'.zip';
		if(name=='mod.json5' && mod.config.name) name=mod.config.name+(mod.config.version?'_'+mod.config.version:'')+'.json5';
		window.saveAs(b, name, {type:'application/'+t[type][0]});
	}
	$scope.uploadSoundpack = mod=>{
		let url = prompt("enter your author password, or leave blank for limited hosting");
		let n = mod.config.name ? '&mod='+btoa(mod.config.name) : '';
		if(!url || url.length<15) url="upload"+(url?'?pass='+url+n:'');
		else if(url.substr(0,4)!='http') return alert('invalid url');
		let b = mod.sounds.asZip({type:'blob'});
		mod.uploadingSoundpack = true;
		fetch(url, {method:"POST", body:b}).then(r => {			
			mod.uploadingSoundpack = false;
			$timeout();
			if(!r.ok) return alert('error uploading');
			r.text().then(r=>{
				console.info('h',r);
				if(r.indexOf('.zip')<0) alert(r);
				else mod.config.soundpack = r;	
				$timeout();
			});			
			
		});
	}
	$scope.uploadMod = mod=>{
		let version = prompt("enter mod version. it's a good time to bump it. if you re-use the same version, there will be cache issues",mod.config.version);
		if(!version) return alert("mod must have a version");
		mod.config.version = version;
		let pass = prompt("enter your author password. if you want your mod listed or your password contact us");
		if(!pass) return alert("no password provided, aborting upload");
		let n = mod.config.name ? '&mod='+btoa(mod.config.name) : '';
		let url="upload?pass="+pass+'@'+version+n;
		let b = mod.asZip({type:'blob'});
		mod.uploadingMod = true;
		fetch(url, {method:"POST", body:b}).then(r => {			
			mod.uploadingMod = false;
			$timeout();
			if(!r.ok) return alert('error uploading');
			r.text().then(r=>{
				alert(r);
			});				
		});
	}
	$scope.mergeNow = mod=>{
		let did=false;		
		try {
			$scope.mods.forEach(m=>{
				if(!m.mergeIt || !m.mergeItOpt.base) return;						
				let opt = m.mergeItOpt;			
				opt.weapons = m.mergeItWp || [];
				opt.wObjects = m.mergeItWo || [];
				if(opt.allWeapons) m.weapons.list.forEach(w=>opt.weapons.push(w.name));
				//if(opt.weapons.length+opt.wObjects.length==0) return;			
				did=true;			
				console.info('merging with base',opt);			
				mod.merge(m,opt);	
			});		
			$scope.mods.forEach(m=>{			
				if(!m.mergeIt || m.mergeItOpt.base) return;						
				let opt = m.mergeItOpt;			
				opt.weapons = m.mergeItWp || [];
				opt.wObjects = m.mergeItWo || [];
				if(opt.allWeapons) m.weapons.list.forEach(w=>opt.weapons.push(w.name));
				if(opt.weapons.length+opt.wObjects.length==0) return;			
				did=true;			
				console.info('merging without base',opt);			
				mod.merge(m,opt);	
			});
		}catch(err){
			console.warn(err);
			alert('an error prevented merging. more info in console');
			return;
		}
		if(!did) return alert('selected nothing to merge');
		mod.loaded=true;
		mod.config.name='MERGED MOD';
		mod.config.colorAnim = [];
		mod.section='wp';
		$scope.newMod();
	}
	$scope.soundSelect = {};
	$scope.objectSelect = {};
	$scope.onSoundSelect = (mod,obj,prop)=>{
		let v = obj[prop];				
		if(v===null || isNaN(v) || v===undefined || v==='') v=-1;		
		obj[prop]=v*1;
		mod.reflect('snd',obj,prop);
		$scope.soundSelect[mod.__id+obj.id+prop]=0;
	}
	$scope.onObjectSelect = (mod,obj,prop)=>{
		let v = obj[prop];				
		if(v===null || isNaN(v) || v===undefined || v==='') v=-1;		
		obj[prop]=v*1;
		mod.reflect(mod.section,obj,prop);
		$scope.objectSelect[mod.__id+obj.id+prop]=0;
	}
	$scope.onModMergeAddWp = m=>{
		let w = m.mergeItWp_;
		m.mergeItWp_=undefined;
		if(!w) return;
		w=w.name;
		if(!m.mergeItWp)m.mergeItWp=[];
		if(m.mergeItWp.indexOf(w)>-1) return;
		m.mergeItWp.push(w);
	}
	$scope.onModMergeAddWo = m=>{
		let w = m.mergeItWo_;
		m.mergeItWo_=undefined;
		if(!w) return;
		w=w.id;
		if(!m.mergeItWo)m.mergeItWo=[];
		if(m.mergeItWo.indexOf(w)>-1) return;
		m.mergeItWo.push(w);
	}
	$scope.isMergeBaseSelected = ()=>{
		for(let i=0;i<$scope.mods.length;i++){
			let m = $scope.mods[i];
			if(m.mergeIt && m.mergeItOpt && m.mergeItOpt.base) return true;
		}
		return false;
	}
	$scope.showHelp = e=>{	
		e = e.currentTarget;
		angular.element(e).parent().toggleClass('show');
	}
	$scope.orderOf = {
		wp:'name',
		w:'id',
		n:'id',
		s:'id',
		c:'id',
		t:'id',
		spr:'id',
		snd:'id'
	}	
	$scope.getWeaponBestSprite = (mod,w)=>{
		let wo = mod.wObjects.list[w.bulletType];
		if(!wo) return;
		if(wo.startFrame>-1) return mod.sprites.list[wo.startFrame];		
		if(wo.colorBullets>0){
			if(!mod.colorBullets) mod.colorBullets = {};
			if(!mod.colorBullets[wo.colorBullets]) mod.colorBullets[wo.colorBullets] = {color:wo.colorBullets};
			return mod.colorBullets[wo.colorBullets];
		}
		if(wo.splinterType>-1){
			let no = mod.nObjects.list[wo.splinterType];
			if(no.startFrame>0) return mod.sprites.list[no.startFrame];
		}			
		
		if(wo.createOnExp>-1){
			let so = mod.sObjects.list[wo.createOnExp];
			if(so.startFrame>-1) return mod.sprites.list[so.startFrame];
		}
	}
	$scope.quickMix = mod=>{
		let k = prompt("how many mods to sample?","5")*1;
		if(isNaN(k)) return;
		let a = $scope.hhtcList.slice();
		a.sort(()=>.5 - Math.random());
		for(let i=0;i<k;i++){
			if(!a[i]) break;
			$scope.quickLoad(mod,{hhtc:a[i],noNewMod:true});			
			mod = $scope.newMod();
		}
		
	};
	class PixelArtEditor {		
		name = "pixel art editor"
		type = "pixelart"
		color = 0
		undolist = []		
		mats = {}
		previews = [];
		constructor(mod,sprite){
			this.id=(Math.random()+'').split('.')[1];
			this.mod=mod;
			this.sprite=sprite;
			this.arg=sprite.name;
			this.colors=[];
			for(let i=0;i<256;i++) this.colors.push(i);
			for(let i=1;i<5;i++) this.previews.push(i);
		}		
		getMat(c){		
			if(!this.mats[c]){
				this.mats[c] = [];
				let m = WLK.materials[c];
				let L = WLK.materialbits;				
				if(m!=0) for(let i=0;i<L.length;i++){
					if(m & 2**i) this.mats[c].push(L[i+1]);
				}					
			}
			return this.mats[c];
		}
		mount(){
			setTimeout(t=>{
			let elem = document.getElementById('ed'+this.id);
			console.info(elem);
			
			let c = this.canvas = document.getElementById('pixelart'+this.id);			
			c.onmousedown = e=>this.onDown(e);
			c.onmousemove = e=>this.onMove(e);
			c.onmouseout = e=>this.onOut(e);
			c.oncontextmenu = e=>this.onCM(e);
			c.onkeypress = e=>this.onPress(e);		
			},100);
		}
		onCM(e){
			try {
				e.preventDefault();
				e.stopPropagation();
				e.stopImmediatePropagation();
			}catch(er){
				console.warn(er);
			}
		}
		onOut(e){
			this.hover=-1;
			this.isDown = false;
			$timeout();
		}
		onPress(e){			
			if (e.code == 'KeyZ' && e.ctrlKey) this.undo();				 
		}
		onDown(e,addundo){			
			this.isDown = true;
			let c = this.canvas, s = this.sprite;
			let x = e.offsetX, y = e.offsetY;			
			x = Math.floor(x/(c.offsetWidth/s.width));
			y = Math.floor(y/(c.offsetHeight/s.height));
			let i = y*s.width+x;
			let v = s.data[i];
			if(e.which === 3 || e.button === 2){ 				
				this.color = v;
				try {
					e.preventDefault();
					e.stopPropagation();
					e.stopImmediatePropagation();
				}catch(er){
					console.warn(er);
				}
			}else{
				let a = this.undolist;
				if(addundo) a = a[a.length-1].list;					
				a.push({i:i,v:v,list:[]});
				s.data[i] = this.color || 0; 
			}			
			$timeout();
		}
		undo(){
			if(!this.undolist.length) return;
			let o = this.undolist.pop();
			o.list.reverse().forEach(oo=>this.sprite.data[oo.i]=oo.v);
			this.sprite.data[o.i]=o.v;
			$timeout();
		}
		onMove(e){			
			if(e.buttons && this.isDown) this.onDown(e,true);			
			let c = this.canvas, s = this.sprite;
			let x = e.offsetX, y = e.offsetY;			
			x = Math.floor(x/(c.offsetWidth/s.width));
			y = Math.floor(y/(c.offsetHeight/s.height));
			this.hover = s.data[y*s.width+x]; 			
			$timeout();			
		}
		selectColor(c){		
			this.color=c;			
		}				
	}
	$scope.pixelArt = (mod,sprite)=>{
		if(!mod.editors) mod.editors = [];
		mod.editors.push(new PixelArtEditor(mod,sprite));
		
	}	
	$scope.mapTools = {		
		deltas:{},		
		save:(map)=>{
			let name = map.id.split('/').pop();
			let s = JSON.stringify(JSON.parse(WLK.stringify(map.settings)), null, "\t");
			let b = new Blob([s], {type: "text/plain;charset=utf-8"});			
			window.saveAs(b, name+'.json', {type:'application/json'});		
		},
		loadJSON:(map)=>{
			askFile('.json', (r,f)=>{								
				f.text().then(r=>{
					console.info('.json',r);				
					map.settings = JSON.parse(r);
					$timeout();				
				});							
				$timeout();
			});
		},
		prepare:(map,mode)=>{
			
			
		},
		onMapDown:(map,ev)=>{
			if(map.refpoint){
				map.refpoint[0] = ev.offsetX;
				map.refpoint[1] = ev.offsetY;
				map.refpoint=undefined;
			}
		},
		onMapMove:(map,ev)=>{			
			map.mouseOver = true;
			map.mouseX = ev.offsetX;
			map.mouseY = ev.offsetY;
		},
		onMapOut:(map,ev)=>{
			map.mouseOver = false;
		},
		addModeSettings:map=>{
			
			if(!map.settings.modes[map.mode]) map.settings.modes[map.mode] = [];
			map.settings.modes[map.mode].push({spawns:[]});
		},
		addSpawn:(map,idx,sp)=>{
			console.info('addSpawn',idx);
			let m = map.settings ? map.settings.modes[map.mode] : map; //modeset ctf/dtf
			if(!m.spawns) m.spawns = [];
			if(!m.spawns[idx]) m.spawns[idx] = [];
			m.spawns[idx].push(sp);
		},
		addHazSpawn:(map,sp)=>{
			let modes = map.settings.modes;
			if(!modes.haz) modes.haz = [];
			if(!modes.haz[0]) modes.haz[0] = [];
			if(!modes.haz[1]) modes.haz[1] = [];
			modes.haz[1].push(sp||[0,0]);
		},
		addZone:(map,a)=>{
			let modes = map.settings.modes;
			if(!modes.haz) modes.haz = [];
			if(!modes.haz[0]) modes.haz[0] = [];
			a = a || [50,50,100,100];
			a.forEach(i=>modes.haz[0].push(i));
		},
		removeZone:(map,d)=>{
			let haz = map.settings.modes.haz
			console.info('removeZone',d,haz);
			for(let i=0;i<4;i++) haz[0].splice(d,1);
		},
		getDeltas:function(a){
			let i = Math.ceil(a.length/4);
			if(!this.deltas[i]){
				this.deltas[i] = [];
				for(j=0;j<i;j++) this.deltas[i].push(j);			
			}
			return this.deltas[i];
		},
		loadLev:function(map){
			var req = new XMLHttpRequest();		
			let url = map.id.substr(0,6)=='https:' ? map.id : 
				'https://gitlab.com/webliero/webliero-maps/-/raw/master/'+map.id+'?inline=true';
			req.open("GET", "levproxy?id="+btoa(url), true);
			req.responseType = "arraybuffer";
			req.onload = ev => {						
				map.sprite = this.lev2sprite(req.response);
				$timeout();
			};
			req.send(null);	
		},
		lev2sprite:function(b){
            if (176400 > b.byteLength || 178448 < b.byteLength)
                throw v.J("Invalid liero level");            
            let a = new WLK_I(new DataView(b),!1);
			let s = new WLK_Sprite(504,350);
            s.data = new Uint8Array(s.width * s.height);
            s.data.set(a.Ga(s.data.length), 0);
			//s.mod = new WLK_Mod();
			s.palette = WLK.palette;
			console.info(s);
			return s;
		}
	}	
	$scope.mapZoneCoords = ['left','top','right','bottom'];
	$scope.mapSpawnTypes = ['green flag','team 1','team 2','blue flag'];
	$scope.mapModes = ['ctf','dtf','pred','haz'];
	$scope.maps = [];
	
	$scope.quickUploadMap = mod=>{
		askFile('.png', (r,f)=>{								
			$scope.quickLoadMap(mod,URL.createObjectURL(f),f.name);
			$timeout();
		});
	}
	$scope.quickLoadMap = (mod,url,name)=>{
		console.info(url);
		let map = mod.quickMap || name;
		mod.quickMap=undefined;
		if(!map) return;
		console.info(map);
		let o = {			
			id: map,
			loaded: true,
			settings: $scope.mapList[map] || {modes:{}}
		}
		if(url) o.url=url;
		if(map.substr(-4).toLowerCase()=='.lev'){
			console.info('LEV MAP');
			o.lev = true;
			o.levurl = $scope.mapTools.loadLev(o);
		}
		$scope.maps.push(o);
		$scope.mapModes.forEach(mode=>{
			if(o.settings.modes[mode]){
				o.mode=mode;
				let M = o.settings.modes;
				if(mode=='dtf' && !Array.isArray(M[mode])){
					console.info('converting to array format',JSON.stringify(M[mode]));
					M[mode] = [M[mode]];
				}
				if(mode=='dtf' && M[mode][0] && Array.isArray(M[mode][0])){
					let a = [];
					if(Array.isArray(M[mode][0][0]) && M[mode][0][0].length==3){
						console.info('converting old triplets',JSON.stringify(M[mode]));
						for(let i=0;i<M[mode].length;i++){
							if(!a[i]) a[i]={spawns:[],order:false,orders:null};
							for(let j=0;j<3;j++){
								a[i].spawns.push([[M[mode][i][j][1],M[mode][i][j][2]]]);
							}
						}						
					}else{
						console.info('converting to array2 format',JSON.stringify(M[mode]));
						M[mode].forEach(aa=>a.push({spawns:aa,order:false,orders:null}));
					}
					M[mode] = a;
				}
				if(mode=='pred' && M[mode][0] && Array.isArray(M[mode][0])){
					console.info('converting to array2 format',JSON.stringify(M[mode]));
					let a = [];
					M[mode].forEach(aa=>a.push({spawns:[[aa]],order:false,orders:null}));
					
					M[mode] = a;
					
				}
				o.modeset=undefined;
				if(M[mode][0]) o.modeset=M[mode][0];
			}
		});
		console.info(o);
	}
	if($scope.wle.attached) return;
	if($scope.offline) return;
	fetch('mods/list.json?'+randomKey).then(r => {
		r.text().then(r=>{
			let q = $scope.quickList = JSON.parse(r);		
			let a = $scope.quickList2 = [];
			for(k in q){
				q[k].id=k;
				a.push(q[k]);
			} 			
			$timeout();
		});
	});	
	fetch('elements.json?'+randomKey).then(r => {
		r.text().then(r=>{
			$scope.elemList = JSON.parse(r);			
			$timeout();
		});
	});	
	fetch('hellhole.json').then(r => {
		r.text().then(r=>{
			$scope.hhtcList = JSON.parse(r);			
			$timeout();
		});
	});	
	fetch('wlmods.json').then(r => {
		r.text().then(r=>{
			$scope.wlmodsList = JSON.parse(r);			
			$timeout();
		});
	});	
	fetch('mapsettings.json').then(r => {
		r.text().then(r=>{
			$scope.mapList = JSON.parse(r);						
			$timeout();
		});
	});	
});

angular.bootstrap(document.getElementById("wledit"), ['wledit']);

function deltaE(rgbA, rgbB) {
  let labA = rgb2lab(rgbA);
  let labB = rgb2lab(rgbB);
  let deltaL = labA[0] - labB[0];
  let deltaA = labA[1] - labB[1];
  let deltaB = labA[2] - labB[2];
  let c1 = Math.sqrt(labA[1] * labA[1] + labA[2] * labA[2]);
  let c2 = Math.sqrt(labB[1] * labB[1] + labB[2] * labB[2]);
  let deltaC = c1 - c2;
  let deltaH = deltaA * deltaA + deltaB * deltaB - deltaC * deltaC;
  deltaH = deltaH < 0 ? 0 : Math.sqrt(deltaH);
  let sc = 1.0 + 0.045 * c1;
  let sh = 1.0 + 0.015 * c1;
  let deltaLKlsl = deltaL / (1.0);
  let deltaCkcsc = deltaC / (sc);
  let deltaHkhsh = deltaH / (sh);
  let i = deltaLKlsl * deltaLKlsl + deltaCkcsc * deltaCkcsc + deltaHkhsh * deltaHkhsh;
  return i < 0 ? 0 : Math.sqrt(i);
}

function rgb2lab(rgb){
  let r = rgb[0] / 255, g = rgb[1] / 255, b = rgb[2] / 255, x, y, z;
  r = (r > 0.04045) ? Math.pow((r + 0.055) / 1.055, 2.4) : r / 12.92;
  g = (g > 0.04045) ? Math.pow((g + 0.055) / 1.055, 2.4) : g / 12.92;
  b = (b > 0.04045) ? Math.pow((b + 0.055) / 1.055, 2.4) : b / 12.92;
  x = (r * 0.4124 + g * 0.3576 + b * 0.1805) / 0.95047;
  y = (r * 0.2126 + g * 0.7152 + b * 0.0722) / 1.00000;
  z = (r * 0.0193 + g * 0.1192 + b * 0.9505) / 1.08883;
  x = (x > 0.008856) ? Math.pow(x, 1/3) : (7.787 * x) + 16/116;
  y = (y > 0.008856) ? Math.pow(y, 1/3) : (7.787 * y) + 16/116;
  z = (z > 0.008856) ? Math.pow(z, 1/3) : (7.787 * z) + 16/116;
  return [(116 * y) - 16, 500 * (x - y), 200 * (y - z)]
}

function simpleColorDistance(a, b) {
    return Math.sqrt(
        Math.pow( ((a[0]) - (b[0])),2 ) +
        Math.pow( ((a[1]) - (b[1])),2 ) +
        Math.pow( ((a[2]) - (b[2])),2 )
    );
};

//FILESAVER:
(function(a,b){if("function"==typeof define&&define.amd)define([],b);else if("undefined"!=typeof exports)b();else{b(),window.FileSaver=a.FileSaver={exports:{}}.exports}})(this,function(){"use strict";function b(a,b){return"undefined"==typeof b?b={autoBom:!1}:"object"!=typeof b&&(console.warn("Deprecated: Expected third argument to be a object"),b={autoBom:!b}),b.autoBom&&/^\s*(?:text\/\S*|application\/xml|\S*\/\S*\+xml)\s*;.*charset\s*=\s*utf-8/i.test(a.type)?new Blob(["\uFEFF",a],{type:a.type}):a}function c(a,b,c){var d=new XMLHttpRequest;d.open("GET",a),d.responseType="blob",d.onload=function(){g(d.response,b,c)},d.onerror=function(){console.error("could not download file")},d.send()}function d(a){var b=new XMLHttpRequest;b.open("HEAD",a,!1);try{b.send()}catch(a){}return 200<=b.status&&299>=b.status}function e(a){try{a.dispatchEvent(new MouseEvent("click"))}catch(c){var b=document.createEvent("MouseEvents");b.initMouseEvent("click",!0,!0,window,0,0,0,80,20,!1,!1,!1,!1,0,null),a.dispatchEvent(b)}}var f="object"==typeof window&&window.window===window?window:"object"==typeof self&&self.self===self?self:"object"==typeof global&&global.global===global?global:void 0,a=/Macintosh/.test(navigator.userAgent)&&/AppleWebKit/.test(navigator.userAgent)&&!/Safari/.test(navigator.userAgent),g=f.saveAs||("object"!=typeof window||window!==f?function(){}:"download"in HTMLAnchorElement.prototype&&!a?function(b,g,h){var i=f.URL||f.webkitURL,j=document.createElement("a");g=g||b.name||"download",j.download=g,j.rel="noopener","string"==typeof b?(j.href=b,j.origin===location.origin?e(j):d(j.href)?c(b,g,h):e(j,j.target="_blank")):(j.href=i.createObjectURL(b),setTimeout(function(){i.revokeObjectURL(j.href)},4E4),setTimeout(function(){e(j)},0))}:"msSaveOrOpenBlob"in navigator?function(f,g,h){if(g=g||f.name||"download","string"!=typeof f)navigator.msSaveOrOpenBlob(b(f,h),g);else if(d(f))c(f,g,h);else{var i=document.createElement("a");i.href=f,i.target="_blank",setTimeout(function(){e(i)})}}:function(b,d,e,g){if(g=g||open("","_blank"),g&&(g.document.title=g.document.body.innerText="downloading..."),"string"==typeof b)return c(b,d,e);var h="application/octet-stream"===b.type,i=/constructor/i.test(f.HTMLElement)||f.safari,j=/CriOS\/[\d]+/.test(navigator.userAgent);if((j||h&&i||a)&&"undefined"!=typeof FileReader){var k=new FileReader;k.onloadend=function(){var a=k.result;a=j?a:a.replace(/^data:[^;]*;/,"data:attachment/file;"),g?g.location.href=a:location=a,g=null},k.readAsDataURL(b)}else{var l=f.URL||f.webkitURL,m=l.createObjectURL(b);g?g.location=m:location.href=m,g=null,setTimeout(function(){l.revokeObjectURL(m)},4E4)}});window.saveAs = f.saveAs=g.saveAs=g,"undefined"!=typeof module&&(module.exports=g)});